<?php

declare(strict_types=1);

namespace Drupal\linkyreplacer;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Url;
use Drupal\entity_route_context\EntityRouteContextRouteHelperInterface;
use Drupal\linky\LinkyInterface;

/**
 * Utility for dealing with Linky entities.
 */
class LinkyEntityUtility implements LinkyEntityUtilityInterface {

  /**
   * LinkyEntityUtility constructor.
   */
  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly LinkyRealmDeterminatorInterface $realmDeterminator,
    protected readonly ConfigFactoryInterface $configFactory,
    protected readonly EntityRouteContextRouteHelperInterface $entityRouteContextRouteHelper,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function normalizeHref(string $href): string {
    if (UrlHelper::isExternal($href) && $this->supportsInternal()) {
      // If a full URI is internal-ish, strip the domain.
      try {
        $url = Url::fromUri($href);
        if ($this->realmDeterminator->isInternal($url->toString())) {
          $parts = parse_url($href);
          // Strip domain and protocol, make it relative.
          $href =
            ($parts['path'] ?? '/') .
            (isset($parts['query']) ? ('?' . $parts['query']) : '') .
            (isset($parts['fragment']) ? ('#' . $parts['fragment']) : '');
        }
      }
      catch (\InvalidArgumentException $e) {
      }
    }

    if (!UrlHelper::isExternal($href) && $this->supportsInternal()) {
      // It is a relative URL.
      if (str_starts_with($href, '/')) {
        $url = Url::fromUri('internal:' . $href);
        if ($url->isRouted()) {
          // Media links should be ignored.
          if (isset($url->getRouteParameters()['media'])) {
            throw new \InvalidArgumentException();
          }

          $routeName = $url->getRouteName();
          $entityTypeId = $this->entityRouteContextRouteHelper->getAllRouteNames()[$routeName] ?? NULL;
          $entityId = $url->getRouteParameters()[$entityTypeId] ?? NULL;
          if ($entityTypeId && $entityId) {
            $query = $url->getOption('query') ? sprintf('?%s', http_build_query($url->getOption('query'))) : NULL;
            $fragment = $url->getOption('fragment') ? sprintf('#%s', $url->getOption('fragment')) : NULL;
            $href = sprintf('entity:%s/%s%s%s', $entityTypeId, $entityId, $query, $fragment);
          }
          else {
            $href = 'internal:' . $url->toString();
          }
        }
        else {
          $href = $url->toUriString();
        }
        // Media links should be ignored, if a /media/id link is used while
        // standalone_url is turned off, the Url would not be considered routed
        // so we need a special check here.
        if (preg_match('/base:media\/\d+/', $href)) {
          throw new \InvalidArgumentException();
        }

        if ($this->preferInternalProtocol() && str_starts_with($href, 'base:')) {
          $href = substr($href, 5);
          $href = 'internal:/' . ltrim($href, '/');
        }
      }
      else {
        throw new \InvalidArgumentException('Relative paths not supported.');
      }
    }

    return $href;
  }

  /**
   * {@inheritdoc}
   */
  public function getLinkyByHref(string $href): ?LinkyInterface {
    // Anchor links are relative, never process.
    if (str_starts_with($href, '#')) {
      throw new \InvalidArgumentException();
    }

    $notUrl = FALSE;
    if (str_starts_with($href, 'tel:')) {
      $notUrl = TRUE;
      if (!$this->supportsTelephone() || strlen($href) <= 9) {
        // It is a 'tel:' link so don't process. Length check is to avoid
        // exceptions from parse_url when the number is less than or equal to
        // 5 characters (so 9 with the tel: prefix).
        // @see https://www.drupal.org/project/drupal/issues/2575577
        throw new \InvalidArgumentException();
      }
    }

    if (str_starts_with($href, 'mailto:')) {
      $notUrl = TRUE;
      if (!$this->supportsMailTo()) {
        // It is a 'mailto:' link so don't process.
        throw new \InvalidArgumentException();
      }
    }

    if (!$notUrl && (!$this->supportsInternal() && $this->realmDeterminator->isInternal($href))) {
      throw new \InvalidArgumentException();
    }

    // It is already a link entity.
    if (preg_match('#/admin/content/linky/(?<entity_id>\d+)#', $href, $matches)) {
      if ($linky = $this->getLinkyById((int) $matches['entity_id'])) {
        return $linky;
      }

      // Linkys should not be made into a Linky.
      throw new \InvalidArgumentException();
    }

    if (!UrlHelper::isExternal($href) && !$this->supportsInternal()) {
      // Internal links cannot be Linky.
      // This check must be *after* the linky path check above.
      throw new \InvalidArgumentException();
    }

    $href = $this->normalizeHref($href);
    return $this->getLinkyByUri($href);
  }

  /**
   * {@inheritdoc}
   */
  public function getLinkyByUri(string $uri): ?LinkyInterface {
    $ids = $this->entityTypeManager->getStorage('linky')->getQuery()
      ->condition('link.uri', $uri)
      ->accessCheck(FALSE)
      ->execute();
    if ($ids) {
      $id = reset($ids);
      return $this->getLinkyById((int) $id);
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function createLinky(string $uri, string $title): LinkyInterface {
    if (\mb_strlen($uri) > 2048) {
      throw new \InvalidArgumentException('URL is too long');
    }
    $uri_parts = \parse_url($uri);
    if ($uri_parts === FALSE) {
      throw new \InvalidArgumentException("The URI '$uri' is malformed.");
    }
    $link = $this->entityTypeManager->getStorage('linky')->create([
      'link' => [
        'uri' => $uri,
        'title' => \mb_substr($title, 0, 255),
      ],
    ]);
    $link->save();
    return $link;
  }

  /**
   * {@inheritdoc}
   */
  public function getLinkyById(int $id): ?LinkyInterface {
    return $this->entityTypeManager->getStorage('linky')->load($id);
  }

  /**
   * Whether HREFs considered internal should be converted to managed links.
   *
   * @return bool
   *   Whether HREFs considered internal should be converted to managed links.
   */
  protected function supportsInternal(): bool {
    return !empty($this->configFactory->get('linkyreplacer.settings')->get('internal'));
  }

  /**
   * Whether email links should be converted to managed links.
   *
   * @return bool
   *   Whether email links should be converted to managed links.
   */
  protected function supportsMailTo(): bool {
    return !empty($this->configFactory->get('linkyreplacer.settings')->get('email'));
  }

  /**
   * Whether telephone links should be converted to managed links.
   *
   * @return bool
   *   Whether telephone links should be converted to managed links.
   */
  protected function supportsTelephone(): bool {
    return !empty($this->configFactory->get('linkyreplacer.settings')->get('telephone'));
  }

  /**
   * Whether to change `base:` style protocol to `internal:`.
   *
   * @return bool
   *   Change protocol for internal links.
   */
  protected function preferInternalProtocol(): bool {
    return !empty($this->configFactory->get('linkyreplacer.settings')->get('internal_prefer_internal'));
  }

}
