<?php

namespace Drupal\tmgmt_contentapi\Services;

use Drupal\Core\Database\Connection;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Access\CsrfTokenGenerator;
use GuzzleHttp\ClientInterface;

/**
 * Service for handling queue operations.
 */
class QueueOperations {
  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The state service for storing temporary data.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The state service for storing temporary data.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * CsrfToken to access controller.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * Constructs a QueueOperations service.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Queue\QueueFactory $queueFactory
   *   The queue object used for processing job items.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrfToken
   *   The CSRF token generator service.
   */
  public function __construct(Connection $database, LoggerChannelFactoryInterface $logger, QueueFactory $queueFactory, StateInterface $state, CsrfTokenGenerator $csrfToken) {
    $this->database = $database;
    $this->logger = $logger;
    $this->queueFactory = $queueFactory;
    $this->state = $state;
    $this->csrfToken = $csrfToken;
  }

  /**
   * Factory method for service instantiation.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   *
   * @return static
   *   The instantiated service.
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('database'),
          $container->get('logger.factory'),
          $container->get('queue'),
          $container->get('state'),
          $container->get('csrf_token')
      );
  }

  /**
   * Logic to start workers for processing the queue.
   *
   * @param string $queue_name
   *   The name of the queue.
   * @param int $batch_size_queue
   *   The batch size for export queue.
   */
  public function processQueue(string $queue_name, $batch_size_queue = 30) {
    // Give option to alter the batch size on the basis of queue name.
    \Drupal::moduleHandler()->alter('tmgmt_contentapi_process_queue', $queue_name, $batch_size_queue);
    // Process the queue via controller.
    $this->processQueueViaController($queue_name, $batch_size_queue);
  }

  /**
   * Process the queue via controller.
   *
   * @param string $queue_name
   *   The name of the queue.
   * @param int $batch_size_queue
   *   The batch size for export queue.
   */
  protected function processQueueViaController(string $queue_name, $batch_size_queue) {
    // Check if current user has permission to run the queue.
    if (!\Drupal::currentUser()->hasPermission('access queue process')) {
      $this->logger->get('TMGMT_CONTENTAPI')->notice('Instant queue execution not permitted, queue will process via cron.');
      return;
    }
    // Generate a CSRF token for the request.
    $csrf_token = $this->csrfToken->get('queue_process_background');
    $this->httpClient = \Drupal::httpClient();
    try {
      $site_url = \Drupal::request()->getSchemeAndHttpHost();
      $endpoint = $site_url . '/tmgmt-contentapi/queue-process-background/' . $queue_name . '/' . $batch_size_queue;
      // Get the session cookie for the current user.
      $session_name = \Drupal::service('session_manager')->getName();
      $session_id = \Drupal::service('session_manager')->getId();
      $cookie = $session_name . '=' . $session_id;
      // Make a non-blocking request.
      $options = [
        'headers' => [
          'X-CSRF-Token' => $csrf_token,
          'Content-Type' => 'application/json',
          'Cookie' => $cookie,
        ],
        'timeout' => 1,
      ];
      $this->httpClient->post($endpoint, $options);
    }
    catch (\Exception $e) {
    }
  }

  /**
   * Get all items from the export queue.
   *
   * @param string $queue_name
   *   The name of the queue.
   * @param string $index_key
   *   The index key to fetch from the data.
   * @param bool $is_manual_check
   *   If TRUE, only items with 'is_manual' set in data will be returned.
   */
  public function getQueueItems(string $queue_name, string $index_key, $is_manual_check = FALSE) {
    $results = $this->database->select('queue', 'q')
      ->fields('q', ['item_id', 'data', 'created'])
      ->condition('name', $queue_name)
      ->execute()->fetchAll();

    $items = [];
    foreach ($results as $record) {
      $data = unserialize($record->data);
      // Check if is_manual_check is true then check if $data['is_manual'] is set then add it to the item array else continue.
      if ($is_manual_check && !isset($data['is_manual'])) {
        continue;
      }
      $items[$record->item_id] = $data[$index_key];
    }
    return $items;
  }

  /**
   * Get number of items exist in queue.
   *
   * @param string $queue_name
   *   Queue name.
   *
   * @return int
   *   Remaining count.
   */
  public function getRemainingItemsFromQueue(string $queue_name) {
    return $remaining_count = $this->database->select('queue', 'q')
      ->condition('name', $queue_name)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

  /**
   * Function to delete item from queue.
   *
   * @param int $item_id
   *   The item id to delete from the queue.
   */
  public function deleteItemFromQueue($item_id) {
    try {
      $this->database->delete('queue')
        ->condition('item_id', $item_id)
        ->execute();
    }
    catch (\Exception $e) {
      $this->logger->get('TMGMT_CONTENTAPI')->error('Error deleting item from queue: @message', ['@message' => $e->getMessage()]);
    }
  }

  /**
   * Function to add item to state variable.
   *
   * @param string $state_key
   *   The state key to store the data.
   * @param mixed $new_item
   *   The new item to add to the state variable.
   * @param mixed $key
   *   The key to use for the new item in the state variable array.
   */
  public function addValueToStatevariable(string $state_key, mixed $new_item, $key = NULL) {

    $current_data = $this->state->get($state_key, []);
    // Ensure the state variable is an array.
    if (!is_array($current_data)) {
      $current_data = [];
    }
    // Add the new item to the array.
    $current_data[$key] = $new_item;
    // Save the updated array back to the state variable.
    $this->state->set($state_key, $current_data);
  }

  /**
   * Function to get array from the state variable.
   *
   * @param string $state_key
   *   The state key to store the data.
   *
   * @return array
   *   The array of items.
   */
  public function getvalueFromStatevariable(string $state_key) {
    return $this->state->get($state_key, []);
  }

  /**
   * Function to delete state variable.
   *
   * @param string $state_key
   *   The state key to delete.
   */
  public function deleteStateVariable(string $state_key) {
    $this->state->delete($state_key);
  }

  /**
   * Stores an API response in the custom database table.
   *
   * @param string $jobkey
   *   The unique key for a batch of items.
   * @param string $item_id
   *   The ID of the item.
   * @param mixed $response
   *   The API response data.
   */
  public function addApiResponse(string $jobkey, string $item_id, mixed $response) {
    // Insert new entry.
    $this->database->insert('tmgmt_capi_response')
      ->fields([
        'jobkey' => $jobkey,
        'item_id' => $item_id,
        'apiresponse' => serialize($response),
        'created' => time(),
      ])
      ->execute();
  }

  /**
   * Retrieves all API responses for a given state key.
   *
   * @param string $jobkey
   *   The state key to fetch responses for.
   *
   * @return array
   *   An array of responses indexed by item_id.
   */
  public function getApiResponsesByJobKey(string $jobkey): array {
    $results = $this->database->select('tmgmt_capi_response', 'a')
      ->fields('a', ['item_id', 'apiresponse'])
      ->condition('jobkey', $jobkey)
      ->execute()
      ->fetchAllKeyed();
    return array_values(array_map(fn($response) => unserialize($response), $results));
  }

  /**
   * Deletes all API responses for a given state key.
   *
   * @param string $jobkey
   *   The state key to delete responses for.
   */
  public function deleteApiResponsesByJobKey(string $jobkey) {
    $this->database->delete('tmgmt_capi_response')
      ->condition('jobkey', $jobkey)
      ->execute();
  }

  /**
   * Get count of items in the table for a given job key.
   */
  public function getApiResponsesCountByJobKey(string $jobkey) {
    return $this->database->select('tmgmt_capi_response', 'a')
      ->condition('jobkey', $jobkey)
      ->countQuery()
      ->execute()
      ->fetchField();
  }

}
