<?php

namespace Drupal\tmgmt_contentapi\Services;

use Drupal\file\FileInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\Entity\File;
use Drupal\File\FileRepositoryInterface;
use Drupal\File\FileUsage\FileUsageInterface;
use Drupal\node\Entity\Node;
use Drupal\tmgmt\JobInterface;
use Drupal\tmgmt_contentapi\Plugin\tmgmt_contentapi\Format\Xliff;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\tmgmt_contentapi\Services\JobHelper;
use ZipArchive;

/**
 * Service handle files related functionalities while job creation.
 *
 * @category Class
 * @package Drupal\tmgmt_contentapi\Services
 * @author Lionbridge team
 */
class ExportJobFiles {
  /**
   * The xliff object.
   *
   * @var \Drupal\tmgmt_contentapi\Plugin\tmgmt_contentapi\Format\Xliff
   */
  protected $exporter;

  /**
   * The filesystem service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The file repository service.
   *
   * @var \Drupal\File\FileRepositoryInterface
   */
  protected $fileRepository;

  /**
   * The file usage service.
   *
   * @var \Drupal\File\FileUsage\FileUsageInterface
   */
  protected $fileUsage;


  /**
   * Contains a reference to the currently being exported job.
   *
   * @var \Drupal\tmgmt\JobInterface
   */
  protected $job;

  /**
   * The clean job label.
   *
   * @var string
   */
  protected $joblabel;

  /**
   * Value for zip job path.
   */
  const ZIP_JOB_PATH = 'zip_job_';

  /**
   * Public file path for sent files.
   */
  const PUBLIC_SENT_FILE_PATH = '://tmgmt_contentapi/LioxSentFiles/';

  /**
  * Value for zip job path.
  */
  const ZIP_REF_PATH = 'zip_ref_';

  /**
   * Public file path for sent files.
   */
  const PUBLIC_REF_FILE_PATH = '://tmgmt_contentapi/LioxRefFiles/';

  /**
   * Zip extenstion.
   */
  const ZIP_EXTENSION = ".zip";

  /**
   * The zip file name with appropriate path.
   *
   * @var string
   */
  protected $zipName;

  /**
   * The zip file name with appropriate path.
   *
   * @var string
   */
  protected $dirNameAllFiles;

  /**
   * All files path.
   *
   * @var string
   */
  public $allFilesPath;

  /**
   * The zip path.
   *
   * @var string
   */
  public $zipPath;

  /**
   * Job helper service.
   *
   * @var \Drupal\tmgmt_contentapi\Services\JobHelper
   */
  protected $jobHelper;

  /**
   * The queue operations.
   *
   * @var \Drupal\tmgmt_contentapi\Services\QueueOperations
   */
  protected $queueOperations;

  /**
   * Consructor.
   *
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   Object for file system service.
   * @param \Drupal\File\FileRepositoryInterface $fileRepository
   *   Object for file repository service.
   * @param \Drupal\File\FileUsage\FileUsageInterface $fileUsage
   *   Object for file usage service.
   * @param \Drupal\tmgmt_contentapi\Services\JobHelper $jobHelper
   *   Service for job helper.
   * @param \Drupal\tmgmt_contentapi\Services\QueueOperations $queueOperations
   *   Object for queue operations service.
   */
  public function __construct(
    FileSystemInterface $fileSystem,
    FileRepositoryInterface $fileRepository,
    FileUsageInterface $fileUsage,
    JobHelper $jobHelper,
    QueueOperations $queueOperations,
  ) {
    $this->fileSystem = $fileSystem;
    $this->fileRepository = $fileRepository;
    $this->fileUsage = $fileUsage;
    $this->jobHelper = $jobHelper;
    $this->queueOperations = $queueOperations;
    // Set default values for variables.
    $this->exporter = new Xliff();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
          $container->get('file.system'),
          $container->get('file.repository'),
          $container->get('file.usage'),
          $container->get('tmgmt_contentapi.job_helper'),
          $container->get('tmgmt_contentapi.queue_operations')
      );
  }

  /**
   * Function to create folder where all exported files will get stored.
   *
   * @return bool
   *   Create a folder and return if its successfuly created or not.
   */
  public function createFolderToStoreExportedFiles() {
    return $this->fileSystem->prepareDirectory($this->allFilesPath, FileSystemInterface::CREATE_DIRECTORY);
  }

  /**
   * Function to export each item of job in same file.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   Job object.
   * @param string $capi_job_id
   *   CAPI job id.
   * @param string $allFilesPath
   *   Path where all files will be stored.
   */
  public function singleExportFileForAllItems($job, $capi_job_id, string $allFilesPath) {
    $labelname = $this->jobHelper->getCleanStringForDirectoryAndFileCreation($this->jobHelper->getJobLabel($job));
    $name = $labelname . "_" . $job->id() . "_all_" . $job->getRemoteSourceLanguage() . '_' . $job->getRemoteTargetLanguage() . '.xlf';
    $jobpath = $allFilesPath . "/" . $name;
    $file = $this->fileRepository->writeData($this->exporter->export($job), $jobpath, FileSystemInterface::EXISTS_REPLACE);
    $this->addFileObjectToTransferArray($file, $capi_job_id, $job->id(), 'all');
  }

  /**
   * Function to spearate Export file for each item of job.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   Object for job.
   * @param string $capi_job_id
   *   CAPI job id.
   * @param int $item_id
   *   Item id.
   * @param string $allFilesPath
   *   Path where all files will be stored.
   */
  public function seperateExportForItem(JobInterface $job, string $capi_job_id, int $item_id, string $allFilesPath) {
    $item = $job->getItems()[$item_id];
    $labelname = $this->jobHelper->getCleanStringForDirectoryAndFileCreation($item->label(), $allFilesPath);
    $name = $labelname . "_" . $job->id() . "_" . $item->id() . "_" . $job->getRemoteSourceLanguage() . '_' . $job->getRemoteTargetLanguage() . '.xlf';
    $itempath = $allFilesPath . "/" . $name;
    $file = $this->fileRepository->writeData($this->exporter->exportItem($item), $itempath, FileSystemInterface::EXISTS_REPLACE);
    $this->addFileObjectToTransferArray($file, $capi_job_id, $job->id(), $item->id());
  }

  /**
   * Function to export each item of job in separate file.
   *
   * @param \Drupal\file\FileInterface $file
   *   File object.
   * @param string $capi_job_id
   *   CAPI job id.
   * @param int $job_id
   *   Job id.
   * @param string $item_id
   *   Item id.
   * @param bool $iszip
   *   If file is zip or not.
   */
  protected function addFileObjectToTransferArray($file, string $capi_job_id, int $job_id, string $item_id, bool $iszip = FALSE) {
    // Add file object to transfer array.
    if ($file instanceof FileInterface) {
      // If file is zip then blank then reset the existing array and set zip file with key "all".
      if ($iszip) {
        // If file is zip, then set the key as 'all'.
        $item_id = 'all';
        // Delete earlier created state variable, Only add zip file to it.
        $this->queueOperations->deleteStateVariable($capi_job_id . '_' . $job_id . '_transfer_files');
      }
      // Add new file object to transfer array.
      $this->queueOperations->addValueToStatevariable($capi_job_id . '_' . $job_id . '_transfer_files', $file, $item_id);
    }
    else {
      // Log error if file is not instance of FileInterface.
      \Drupal::logger('tmgmt_contentapi')->error('File object is not instance of FileInterface: @file', ['@file' => print_r($file, TRUE)]);
      // If file is not instance of FileInterface, then throw exception.
      throw new \Exception("File object is not instance of FileInterface.");
    }
  }

  /**
   * Function to zip exported file.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   Object for job.
   * @param array $file_array_exported_files
   *   Array of exported files.
   * @param string $zip_path
   *   Path where zip file will be stored.
   */
  public function zipExportedFiles($job, $file_array_exported_files, string $zip_path) {
    // Zip the exported files.
    if (empty($file_array_exported_files)) {
      return;
    }
    $ziparchive = new \ZipArchive();
    $openresult = $ziparchive->open($this->fileSystem->realpath(uri: $zip_path), ZIPARCHIVE::CREATE | ZIPARCHIVE::OVERWRITE);
    $zipcloseresult = FALSE;
    $addedFiles = 0;
    if ($openresult) {
      foreach ($file_array_exported_files as $tempfile) {
        if ($tempfile == NULL || $tempfile->getFileUri() == NULL || strlen($tempfile->getFileUri()) == 0) {
          continue;
        }
        if ($this->fileSystem->realpath($tempfile->getFileUri()) == NULL) {
          continue;
        }
        $ziparchive->addFile($this->fileSystem->realpath($tempfile->getFileUri()), $tempfile->getFilename());
        $addedFiles = $addedFiles + 1;
      }
      $zipcloseresult = FALSE;
      if ($addedFiles > 0) {
        $zipcloseresult = $ziparchive->close();
      }
      if ($zipcloseresult) {
        $zipfileobj = $this->jobHelper->createFileObject($zip_path);
        $this->fileUsage->add($zipfileobj, 'tmgmt_contentapi', 'tmgmt_job', $job->id());
      }
    }
  }

  /**
   * Check condition before moving ahead to job creation.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   Object for job.
   */
  public function setConditions(JobInterface $job) {
    // Perform action only if capi setting task set to trans.
    if ($job->getSetting('capi-settings')["task"] != "trans") {
      return;
    }
    // Create folder to store exported files else throw exception.
    if (!$this->createFolderToStoreExportedFiles()) {
      throw new \Exception("Could not create directory for export: " . $this->allFilesPath);
    }
  }

  /**
   * Function to prepare zip file to transfer if "Transfer all files as zip" option checked.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   Object for job.
   * @param string $capi_job_id
   *   CAPI job id.
   * @param string|null $zipPath
   *   Path where zip file will be stored.
   */
  public function finalZipArrayOfFilesToTransfer(JobInterface $job, $capi_job_id, $zipPath = NULL) {
    // If exported files are transfered as zip, delete org. exports as already in the zip.
    if ($job->getTranslator()->getSetting('transfer-settings')) {
      // Add zip to transfer array.
      if (!file_exists($zipPath)) {
        return;
      }
      $this->addFileObjectToTransferArray($this->jobHelper->createFileObject($zipPath), $capi_job_id, $job->id(), 'all', TRUE);

    }
  }

  /**
   * Function to prepare files to transfer.
   *
   * @param \Drupal\tmgmt\JobInterface $job
   *   Object for job.
   * @param bool $is_translation_memory
   *   Check if its translation memory.
   */
  public function jobFilesToTransfer(JobInterface $job, $capi_jpob_id, bool $is_translation_memory = FALSE) {
    // Step 1: Initiate variables and set path.
    $this->joblabel = $capi_jpob_id;
    // If is translation memory set then change the directory name.
    if ($is_translation_memory) {
      $this->dirNameAllFiles = $this->joblabel . '_tmupdate_' . $job->id() . "_" . $job->getRemoteSourceLanguage() . "_" . $job->getRemoteTargetLanguage() . "_tm";
    }
    else {
      $this->dirNameAllFiles = $this->joblabel . '_' . $job->id() . "_" . $job->getRemoteSourceLanguage() . "_" . $job->getRemoteTargetLanguage();
    }

    $this->zipName = self::ZIP_JOB_PATH . $this->dirNameAllFiles . self::ZIP_EXTENSION;
    $this->allFilesPath = $job->getSetting('scheme') . self::PUBLIC_SENT_FILE_PATH . $this->dirNameAllFiles;
    $this->zipPath = $this->allFilesPath . "/" . $this->zipName;
    // Step 2: Set condition as per the settings else return.
    $this->setConditions($job);
  }

  /**
   * Function to delete all files once job submitted.
   */
  public function deleteProcessedFiles($file_array) {
    if (empty($file_array)) {
      return;
    }
    foreach ($file_array as $tempfile) {
      $this->fileSystem->delete($tempfile->getFileUri());
    }
  }

}
