<?php

declare(strict_types=1);

namespace Drupal\listing_page;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\ContentEntityTypeInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\views\Views;
use Drupal\Core\Form\FormState;

/**
 * Helper service for various listing page features.
 */
final class Helper {

  /**
   * Object constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entityTypeBundleInfo
   *   The entity type bundle info service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   *   The form builder service.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly FormBuilderInterface $formBuilder,
  ) {}

  /**
   * Checks if given entity is a listing entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   *
   * @return bool
   *   True if this is a listing entity.
   */
  public function isListingEntity(ContentEntityInterface $entity) {
    return (bool) $this->getEntityViewsField($entity);
  }

  /**
   * Returns the field that holds the views ID for the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to check.
   *
   * @return string|bool
   *   The views field name, false otherwise.
   */
  public function getEntityViewsField(ContentEntityInterface $entity) {
    foreach ($entity->getFields() as $field_name => $field) {
      if ('entity_listing_info' == $field->getFieldDefinition()->getType()) {
        return $field_name;
      }
    }
    return FALSE;
  }

  /**
   * Returns an array of available content entity types.
   *
   * @return array
   *   The array of available content entity types.
   */
  public function getContentEntityTypes() {
    $entity_types_bundles = &drupal_static('listing_page.entity_types');
    if (isset($entity_types_bundles)) {
      return $entity_types_bundles;
    }
    $entity_types_bundles = [];
    $entity_types = $this->entityTypeManager->getDefinitions();

    // List only Content entity types that are fildable.
    $entity_types = array_filter($entity_types, function ($entity_type) {
      return ($entity_type instanceof ContentEntityTypeInterface && $entity_type->get('field_ui_base_route'));
    });

    foreach ($entity_types as $name => $entity_type) {
      $entity_types_bundles[$name] = [
        'label' => $entity_type->getLabel(),
        'bundles' => $this->entityTypeBundleInfo->getBundleInfo($name),
      ];
    }

    return $entity_types_bundles;
  }

  /**
   * Returns an array of defined entity listing types and bundles.
   *
   * @return array
   *   The array of defined entity listing types and bundles.
   */
  public function getListingEntitiesDefinitions() {
    $listing_entities_definitions = &drupal_static('listing_page.definitions');
    if (isset($listing_entities_definitions)) {
      return $listing_entities_definitions;
    }
    $listing_entities_definitions = [];
    $entity_types = $this->getContentEntityTypes();
    foreach ($entity_types as $entity_type_id => $data) {
      foreach ($data['bundles'] as $bundle => $value) {
        $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
        foreach ($fields as $field_name => $field) {
          if ('entity_listing_info' === $field->getFieldStorageDefinition()->getType()) {
            $listing_entities_definitions[$entity_type_id][$bundle] = $field_name;
          }
        }
      }
    }

    return $listing_entities_definitions;
  }

  /**
   * Returns an array containing all the created listing entities.
   *
   * This array is keyed by entity_type_id and bundle.
   *
   * @return array
   *   Available listing entities.
   */
  public function getListingEntities() {
    $list = &drupal_static('listing_page.entities');
    if (isset($list)) {
      return $list;
    }
    $list = [];
    $listing_entity_types = $this->getListingEntitiesDefinitions();
    foreach ($listing_entity_types as $entity_type_id => $bundles) {
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      foreach ($bundles as $bundle => $listing_field_name) {
        // Load listing entities.
        $listing_entities = $this->entityTypeManager->getStorage($entity_type_id)->loadByProperties([
          $entity_type->getKey('bundle') => $bundle,
        ]);

        // Remove listing entities with empty "Content type" sub field.
        // $listing_entities = array_filter($listing_entities, function (ContentEntityInterface $entity) use ($listing_field_name) {
        //   return !$entity->get($listing_field_name)->isEmpty() &&
        //     !empty($entity->get($listing_field_name)->first()->getEntityTypes());
        // });
        $list[$entity_type_id][$bundle] = [
          'entities' => $listing_entities,
          'field_name' => $listing_field_name,
        ];
      }
    }

    return $list;
  }

  /**
   * Returns the listing entity for the given entity.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to get the listing entity (parent) for.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|bool
   *   The listing entity for the given entity.
   */
  public function getListingEntityForEntity(ContentEntityInterface $entity) {
    $list = $this->getListingEntities();
    $context = [
      'entity' => $entity,
    ];
    $this->moduleHandler->alter('listing_entities', $list, $context);
    $result = FALSE;
    foreach ($list as $bundles) {
      foreach ($bundles as $array) {
        /** @var Drupal\Core\Entity\ContentEntityInterface $listing_entity */
        foreach ($array['entities'] as $listing_entity) {
          $key = $entity->getEntityTypeId() . ':' . $entity->bundle();
          if (!$listing_entity->get($array['field_name'])->isEmpty()) {
            if(!empty($listing_entity->get($array['field_name'])->first()->getEntityTypes())) {
              if(in_array($key, $listing_entity->get($array['field_name'])->first()->getEntityTypes())) {
                $result = $listing_entity;
                break(3);
              }
            }
          }
        }
      }
    }
    return $result;
  }

  /**
   * Returns a listing entity for the given entity_type_id and bundle.
   *
   * The first found entity is returned.
   *
   * @todo Maybe introduce some kind of negotiator de define wich
   * entity we should use when several candidates are available?
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|bool
   *   The listing entity for the provided entity_type_id and bundle.
   *   False if none is found.
   */
  public function getListingEntityForBundle(string $entity_type_id, string $bundle) {
    $list = $this->getListingEntities();
    $context = [
      'entity_type_id' => $entity_type_id,
      'bundle' => $bundle,
    ];

    $this->moduleHandler->alter('listing_entities', $list, $context);

    $result = FALSE;
    foreach ($list as $bundles) {
      foreach ($bundles as $array) {
        /** @var Drupal\Core\Entity\ContentEntityInterface $listing_entity */
        foreach ($array['entities'] as $listing_entity) {
          $key = $entity_type_id . ':' . $bundle;
          if (!$listing_entity->get($array['field_name'])->isEmpty()) {
            if(!empty($listing_entity->get($array['field_name'])->first()->getEntityTypes())) {
              if(in_array($key, $listing_entity->get($array['field_name'])->first()->getEntityTypes())) {
                $result = $listing_entity;
                break(3);
              }
            }
          }
        }
      }
    }

    return $result;
  }

  /**
   * Returns a listing entity for the given views and display ids.
   *
   * The first found entity is returned.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface|bool
   *   The listing entity for the provided entity_type_id and bundle.
   *   False if none is found.
   */
  public function getListingEntityByViews(string $views_id, string $display_id) {
    $list = $this->getListingEntities();
    $result = FALSE;
    foreach ($list as $bundles) {
      foreach ($bundles as $array) {
        /** @var Drupal\Core\Entity\ContentEntityInterface $listing_entity */
        foreach ($array['entities'] as $listing_entity) {
          if (
            !$listing_entity->get($array['field_name'])->isEmpty() &&
            ($listing_entity->get($array['field_name'])->first()->getViews() == "{$views_id}:{$display_id}")
          ) {
            $result = $listing_entity;
            break(3);
          }
        }
      }
    }

    return $result;
  }

  /**
   * Returns the exposed form of a views.
   *
   * Takes care of setting the #action attribute.
   *
   * @param string $views_id
   *   The views id.
   * @param string $views_display_id
   *   The views display id.
   * @param string $views_main_display_id
   *   The main views display id.
   *
   * @return array
   *   A render array representing the search form.
   */
  public function getListingExposedForm($views_id, $views_display_id = 'form', $views_main_display_id = 'listing_page') {
    $view = Views::getView($views_id);
    if (empty($view)) {
      return [];
    }

    $view->setDisplay($views_display_id);
    $view->initHandlers();

    if ($listing_entity = $this->getListingEntityByViews($views_id, $views_main_display_id)) {
      $view->override_url = $listing_entity->toUrl();
    }

    $form_state = (new FormState())
      ->setStorage([
        'view' => $view,
        'display' => &$view->display_handler->display,
        'rerender' => TRUE,
      ])
      ->setMethod('get')
      ->setAlwaysProcess()
      ->disableRedirect();
    $form_state->set('rerender', NULL);
    $form = $this->formBuilder
      ->buildForm('\Drupal\views\Form\ViewsExposedForm', $form_state);

    return $form;
  }

}
