<?php

namespace Drupal\listing_page\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\listing_page\Helper;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\views\Views;

/**
 * Hook implementations for token.
 */
class EntityHooks {

  public function __construct(
    protected readonly Helper $listingHelper,
  ) {}

  /**
   * Implements hook_entity_update().
   *
   * Invalidate listing_entity_list & listing_entity_list:ENTITY_TYPE_ID:BUNDLE
   * tags when a listing entity is created, updated or deleted.
   */
  #[Hook('entity_insert')]
  #[Hook('entity_update')]
  #[Hook('entity_delete')]
  public function invalidateListingEntitiesCacheTags(EntityInterface $entity) {
    // Only act on content entities.
    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    // Only act on listing entities.
    if (!$this->listingHelper->isListingEntity($entity)) {
      return;
    }

    Cache::invalidateTags([
      'listing_entity_list',
      'listing_entity_list:' . $entity->getEntityTypeId() . ':' . $entity->bundle(),
    ]);
  }

  /**
   * Implements hook_entity_access().
   *
   * Checks if the embeded views is accessible by the cirrent user.
   */
  #[Hook('entity_access')]
  public function entityAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    if (
      $operation === 'view' &&
      $entity instanceof ContentEntityInterface &&
      $this->listingHelper->isListingEntity($entity)
    ) {
      $field_name = $this->listingHelper->getEntityViewsField($entity);
      if (!$entity->get($field_name)->isEmpty()) {
        $views_id = $entity->get($field_name)->first()->getViews();
        [$views_id, $display_id] = explode(':', $views_id);
        $view = Views::getView($views_id);
        if (!$view || !$view->access($display_id)) {
          return AccessResult::forbidden('No access to views');
        }
      }
    }
    return AccessResult::neutral();
  }

}
