<?php

declare(strict_types=1);

namespace Drupal\listing_page\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\listing_page\Helper;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Form\FormStateInterface;

/**
 * Hook implementations for listing_page (Fields).
 */
class FieldHooks {

  use StringTranslationTrait;

  public function __construct(
    private readonly EntityFieldManagerInterface $entityFieldManager,
    protected readonly Helper $listingHelper,
  ) {}

  /**
   * Implements hook_entity_extra_field_info_alter().
   *
   * Define "Listing views results" extra field on listing entities.
   */
  #[Hook('entity_extra_field_info_alter')]
  public function attachListingFields(&$info): void {
    foreach ($info as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle => $extra_fields) {
        $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
        foreach ($fields as $field) {
          if ('entity_listing_info' === $field->getFieldStorageDefinition()->getType()) {
            $info[$entity_type_id][$bundle]['display']['listing_page_views'] = [
              'label' => 'Views results',
              'description' => $this->t('Listing views results'),
              'weight' => 100,
              'visible' => TRUE,
            ];
            break;
          }
        }
      }
    }
  }

  /**
   * Implements hook_entity_view().
   *
   * When using the entity_listing_info_complex formatter, we pass the listing
   * entity build to views. This is usefull for theming purposes :
   * Listing entity build fields will be available in the views-view.html.twig
   * template.
   */
  #[Hook('entity_view')]
  public function computeListingFields(array &$build, ContentEntityInterface $entity, EntityViewDisplayInterface $display, $view_mode): void {
    if ($field_name = $this->listingHelper->getEntityViewsField($entity)) {
      if (!$entity->get($field_name)->isEmpty() && ($listing_field_display = $display->getComponent($field_name))) {
        if ('entity_listing_info_complex' == $listing_field_display['type']) {
          $build[$field_name][0]['#listing_entity_build'] = [];
          foreach (Element::children($build, TRUE) as $key) {
            // Skip the actual listing field.
            if ($field_name == $key) {
              continue;
            }
            $build[$field_name][0]['#listing_entity_build'][$key] = $build[$key];
          }
          $build[$field_name][0]['#listing_entity'] = $entity->getEntityTypeId() . ':' . $entity->id();
        }
      }
    }
  }

  /**
   * Implements hook_entity_extra_field_info_alter().
   *
   * Define "Listing page url" extra field on entities.
   */
  #[Hook('entity_extra_field_info_alter')]
  public function attachListedFields(&$info): void {
    foreach ($info as $entity_type_id => $bundles) {
      foreach ($bundles as $bundle => $extra_fields) {
        $fields = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);
        $is_listing = FALSE;
        foreach ($fields as $field) {
          if ('entity_listing_info' === $field->getFieldStorageDefinition()->getType()) {
            $is_listing = TRUE;
            break;
          }
        }
        if (!$is_listing) {
          $info[$entity_type_id][$bundle]['display']['listing_page_url'] = [
            'label' => $this->t('Listing page'),
            'description' => $this->t('Listing page url'),
            'weight' => 100,
            'visible' => FALSE,
          ];
        }
      }
    }
  }

  /**
   * Implements hook_entity_view().
   *
   * Compute "Listing page url" extra field on entities.
   */
  #[Hook('entity_view')]
  public function computeListedFields(array &$build, ContentEntityInterface $entity, EntityViewDisplayInterface $display, $view_mode): void {
    if ($display->getComponent('listing_page_url')) {
      if ($listing_entity = $this->listingHelper->getListingEntityForEntity($entity)) {
        $build['listing_page_url'] = [
          '#type' => 'link',
          '#title' => $this->t('Back to listing'),
          '#url' => $listing_entity->toUrl(),
        ];
      }
    }
  }

  /**
   * Implements hook_field_widget_complete_form_alter().
   *
   * Alter pathauto widget : prevent automatically generate new paths on
   * listing entity updates.
   * The alias probably used to compute aliases for the listed entities.
   */
  #[Hook('field_widget_complete_form_alter')]
  public function fieldWidgetCompleteFormAlter(&$field_widget_complete_form, FormStateInterface $form_state, $context): void {
    if ('path' == $context['widget']->getPluginId()) {
      if (array_key_exists('pathauto', $field_widget_complete_form['widget'][0])) {
        $entity = $form_state->getFormObject()->getEntity();
        if ($this->listingHelper->isListingEntity($entity)) {
          $field_widget_complete_form['widget'][0]['pathauto']['#default_value'] = 0;
        }
      }
    }
  }

}
