<?php

namespace Drupal\listing_page\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityChangedInterface;

/**
 * Hook implementations for listing_page.
 */
class ListingPageHooks {

  public function __construct(
    protected readonly LanguageManagerInterface $languageManager,
  ) {}

  #[Hook('listing_entities_alter')]
  /**
   * Remove entities that the current user can't access.
   *
   * @param array $list
   *   Listing entities.
   */
  public function checkAccess(&$list) {
    foreach ($list as &$bundles) {
      foreach ($bundles as &$bundle_data) {
        // $bundle_data['entities'] = array_filter($bundle_data['entities'], function (ContentEntityInterface $entity) {
        //   return $entity->access('view');
        // });
      }
    }
  }

  #[Hook('listing_entities_alter')]
  /**
   * Exclude unpublished entities (which have a "Published" state).
   *
   * @param array $list
   *   The list of entities grouped by bundles.
   * @param array $context
   *   The context containing the current entity.
   */
  public function filterPublishedEntities(array &$list, array $context): void {
    $interface = EntityPublishedInterface::class;
    foreach ($list as &$bundles) {
      foreach ($bundles as &$bundle_data) {
        $bundle_data['entities'] = array_filter($bundle_data['entities'], function (ContentEntityInterface $entity) use ($interface) {
          if ($entity->getEntityType()->entityClassImplements($interface) && !$entity->isPublished()) {
            return FALSE;
          }
          return TRUE;
        });
      }
    }
  }

  #[Hook('listing_entities_alter')]
  /**
   * Match entities to the language of the target entity.
   *
   * @param array $list
   *   The list of entities grouped by bundles.
   * @param array $context
   *   The context containing the current entity.
   */
  public function matchEntityLanguage(array &$list, array $context): void {
    // Ensure the context contains a valid entity with a language.
    if (empty($context['entity']) || !$context['entity'] instanceof ContentEntityInterface) {
      return;
    }

    $langcode = $context['entity']->language()->getId();
    if (!$langcode) {
      return;
    }

    // Iterate through the list and match entities to the current language.
    foreach ($list as &$bundles) {
      foreach ($bundles as &$bundle_data) {
        $bundle_data['entities'] = array_map(function (ContentEntityInterface $entity) use ($langcode) {
          return $entity->hasTranslation($langcode) ? $entity->getTranslation($langcode) : $entity;
        }, $bundle_data['entities']);
      }
    }
  }

  #[Hook('listing_entities_alter')]
  /**
   * Match entities to the current language.
   *
   * @param array $list
   *   The list of entities grouped by bundles.
   * @param array $context
   *   The context containing the current entity.
   */
  public function matchCurrentLanguage(array &$list, array $context): void {
    // Ensure the context contains a valid entity with a language.
    if (empty($context['entity_type_id']) || empty($context['bundle'])) {
      return;
    }

    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    if (!$langcode) {
      return;
    }

    // Iterate through the list and match entities to the current language.
    foreach ($list as &$bundles) {
      foreach ($bundles as &$bundle_data) {
        $bundle_data['entities'] = array_map(function (ContentEntityInterface $entity) use ($langcode) {
          return $entity->hasTranslation($langcode) ? $entity->getTranslation($langcode) : $entity;
        }, $bundle_data['entities']);
      }
    }
  }

  #[Hook('listing_entities_alter')]
  /**
   * Sort entities by changed date DESC.
   *
   * @param array $list
   *   The list of entities grouped by bundles.
   * @param array $context
   *   The context containing the current entity.
   */
  public function sortByChangedDate(array &$list, array $context): void {
    $interface = EntityChangedInterface::class;
    foreach ($list as &$bundles) {
      foreach ($bundles as &$bundle_data) {
        if (($entity = current($bundle_data['entities'])) && $entity->getEntityType()->entityClassImplements($interface)) {
          uasort($bundle_data['entities'], function (EntityChangedInterface $a, EntityChangedInterface $b) {
            return $a->getChangedTime() - $b->getChangedTime();
          });
        }
      }
    }
  }

}
