<?php

namespace Drupal\listing_page\Hook;

use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\listing_page\Helper;
use Drupal\Core\Utility\Token;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Hook implementations for token.
 */
class TokenHooks {

  use StringTranslationTrait;

  public function __construct(
    protected readonly Helper $listingHelper,
    protected readonly Token $tokenService,
  ) {}

  /**
   * Implements hook_token_info().
   */
  #[Hook('token_info')]
  public function tokenInfo() {
    $entity_types = $this->listingHelper->getContentEntityTypes();
    // Define tokens like [node:related_listing_entity:*].
    foreach (array_keys($entity_types) as $entity_type_id) {
      $type = $this->getTokenType($entity_type_id);
      $tokens[$type] = [
        'related_listing_entity' => [
          'name' => $this->t("The related listing entity"),
          'description' => $this->t("The related listing entity"),
          'type' => $type,
        ],
      ];
    }

    // Define tokens like [site:listing_page:node:news:*].
    $tokens['site'] = [
      'listing_page' => [
        'name' => $this->t("Listing page"),
        'description' => $this->t("Get listing properties title, url, id"),
        'dynamic' => TRUE,
      ],
    ];

    return [
      'tokens' => $tokens,
    ];

  }

  /**
   * Implements hook_tokens().
   */
  #[Hook('tokens')]
  public function tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata) {
    $replacements = [];
    $entity_types = $this->listingHelper->getContentEntityTypes();
    // Per entity tokens : [node:related_listing_entity:*].
    foreach (array_keys($entity_types) as $entity_type_id) {
      $token_type = $this->getTokenType($entity_type_id);
      if ($type == $token_type && !empty($data[$token_type])) {
        $entity = $data[$token_type];
        if (
          ($parent_tokens = $this->tokenService->findWithPrefix($tokens, 'related_listing_entity')) &&
          ($listing_entity = $this->listingHelper->getListingEntityForEntity($entity))
        ) {
          // Expire token when a new listing entity is created.
          $bubbleable_metadata->addCacheTags([
            'listing_entity_list',
            'listing_entity_list:' . $listing_entity->getEntityTypeId() . ':' . $listing_entity->bundle(),
          ]);
          $listing_entity_token_type = $this->getTokenType($listing_entity->getEntityTypeId());
          $replacements += $this->tokenService->generate($listing_entity_token_type, $parent_tokens, [$listing_entity_token_type => $listing_entity], $options, $bubbleable_metadata);
        }
      }
    }

    // Global tokens : [site:listing_page:node:news:*].
    if ($type == 'site') {
      $listing_page_tokens = $this->tokenService->findWithPrefix($tokens, 'listing_page');
      if (!empty($listing_page_tokens)) {
        foreach ($listing_page_tokens as $token => $original) {
          // Ensure $token is composed of exactly 3 parts separated by ':'.
          // Than extract entity type id and bundle from token.
          if (substr_count($token, ':') < 2) {
            continue;
          }
          [$entity_type_id, $bundle, $token] = explode(':', $token, 3);

          // Get the listing_entity that corresponds to
          // the entity type id and bundle.
          $listing_entity = $this->listingHelper->getListingEntityForBundle($entity_type_id, $bundle);

          // If a $listing_entity is found, chain its tokens.
          if (!empty($listing_entity)) {
            // Expire token when a new listing entity is created.
            $listing_entity_type_id = $listing_entity->getEntityTypeId();
            $listing_entity_bundle = $listing_entity->bundle();
            $bubbleable_metadata->addCacheTags([
              'listing_entity_list',
              'listing_entity_list:' . $listing_entity_type_id . ':' . $listing_entity_bundle,
            ]);

            $token_type = $this->getTokenType($listing_entity_type_id);
            $token_chained = [$token => "[{$token_type}:{$token}]"];
            $replacements_chained = $this->tokenService->generate($token_type, $token_chained, [$token_type => $listing_entity], $options, $bubbleable_metadata);
            if (!empty($replacements_chained) && isset($replacements_chained[$token_chained[$token]])) {
              $replacements[$original] = $replacements_chained[$token_chained[$token]];
            }
          }
        }
      }
    }

    return $replacements;
  }

  /**
   * Returns the proper token type for the given entity type id.
   *
   * Because some token types to do not match their entity type names,
   * we have to map them to the proper type.
   *
   * @param string $entity_type_id
   *   The entity type id.
   *
   * @see token_entity_type_alter()
   *
   * @return string
   *   Returns the taxonomy token type string.
   */
  protected function getTokenType($entity_type_id) {
    // Fill in default token types for entities.
    switch ($entity_type_id) {
      case 'taxonomy_term':
      case 'taxonomy_vocabulary':
        // Stupid taxonomy token types indeed...
        $type = str_replace('taxonomy_', '', $entity_type_id);
        break;

      default:
        // By default the token type is the same as the entity type.
        $type = $entity_type_id;
        break;
    }
    return $type;
  }

}
