<?php

declare(strict_types=1);

namespace Drupal\listing_page\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\views\ViewExecutable;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Hook implementations for listing_page (Views).
 */
class ViewsHooks {

  public function __construct(
    protected readonly EntityTypeManagerInterface $entityTypeManager,
    protected readonly EntityDisplayRepositoryInterface $entityDisplayRepository,
    protected readonly ModuleHandlerInterface $moduleHandler,
    protected readonly RouteMatchInterface $routeMatch,
  ) {}

  #[Hook('form_alter')]
  /**
   * Implements hook_form_alter()
   *
   * Add #view property to form this is needed for theming.
   *
   * @see listing_page_preprocess_views_exposed_form()
   */
  public function viewsExposedFormAlter(&$form, FormStateInterface $form_state, $form_id): void {
    if ('views_exposed_form' == $form_id) {
      $view = $form_state->getStorage()['view'];
      $form['#view'] = $view;
    }
  }

  #[Hook('views_post_render')]
  /**
   * Implements hook_views_pre_render()
   *
   * When views ajax is enabled, the listing entity is lost.
   * This is a hack to retrieve it from request.
   *
   * @todo find a more reliable way to do this.
   */
  public function viewsPostRender(ViewExecutable $view): void {
    if ($view->display_handler->getPluginId() == 'listing_page') {

      // On ajax calls the listing_entity_build is lost.
      // Let's use drupalSettings to pass it back and forth between
      // drupal and the browser.
      if (!empty($view->element['#listing_entity'])) {
        $view->element['#attached']['drupalSettings']['views']['ajaxViews']['views_dom_id:' . $view->dom_id]['listing_entity'] = $view->element['#listing_entity'];
      }
      // This is likely an ajax call, retrieve the listing entity from
      // drupalSettings (which is passed in the request)
      elseif ($query_str = parse_url($view->getRequest()->getRequestUri(), PHP_URL_QUERY)) {
        parse_str($query_str, $query_params);
        if (!empty($query_params['listing_entity'])) {
          // Make sure listing_entity is always passed to drupalSettings.
          $view->element['#attached']['drupalSettings']['views']['ajaxViews']['views_dom_id:' . $view->dom_id]['listing_entity'] = $query_params['listing_entity'];

          [$entity_type_id, $id] = explode(':', $query_params['listing_entity']);
          if ($entity = $this->entityTypeManager->getStorage($entity_type_id)->load($id)) {
            // Set the current display plugin.
            // needed later for theme suggesting.
            $view->element['#display_plugin_id'] = $view->getDisplay()->getPluginId();

            // Get fields from the default display.
            $build = [];
            $view_display = $this->entityDisplayRepository->getViewDisplay($entity->getEntityTypeId(), $entity->bundle(), 'default');
            $fields = $view_display->get('content');
            foreach ($fields as $field_name => $field) {
              if (empty($field['type'])) {
                unset($fields[$field_name]);
              }
              else {
                $build[$field_name] = $entity->get($field_name)->view($field);
              }
            }

            // Apply entity view hooks on the build,
            // so other modules hooks get processed.
            $this->moduleHandler->invokeAll('entity_view', [&$build, $entity, $view_display, 'full']);

            // Pass the build to preprocess.
            $view->element['#listing_entity_build'] = $build;
          }

        }
      }
    }
  }

}
