<?php

declare(strict_types=1);

namespace Drupal\listing_page\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\MapDataDefinition;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines the 'entity_listing_info' field type.
 */
#[FieldType(
  id: 'entity_listing_info',
  label: new TranslatableMarkup('Entity listing information'),
  description: new TranslatableMarkup('Field to store listing information (Views id & entity type)'),
  default_widget: 'entity_listing_info_default',
  default_formatter: 'entity_listing_info_default',
  category: "listing",
)]
final class EntityListingInfoItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {

    $columns = [
      'views' => [
        'type' => 'varchar',
        'not null' => TRUE,
        'description' => 'The listing views & display',
        'length' => 255,
      ],
      'entity_types' => [
        'type' => 'blob',
        'size' => 'big',
        'serialize' => TRUE,
        'not null' => FALSE,
        'description' => 'The displayed entity type & bundle',
      ],
    ];

    $schema = [
      'columns' => $columns,
      'indexes' => [
        'views' => ['views'],
        'entity_types' => ['entity_types'],
      ],
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'multiple_bundles' => FALSE,
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = [];

    $element['multiple_bundles'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Multiple content types listing'),
      '#description' => $this->t('Entities of this type may list multiple content types.'),
      '#default_value' => $this->getSetting('multiple_bundles'),
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {

    $properties['views'] = DataDefinition::create('string')
      ->setLabel(t('The listing views & display'))
      ->setRequired(TRUE);
    $properties['entity_types'] = MapDataDefinition::create()
      ->setLabel(t('The displayed entity type & bundle'))
      ->setRequired(FALSE);

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function mainPropertyName() {
    return 'views';
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    return match ($this->get('views')->getValue()) {
      NULL, '' => TRUE,
      default => FALSE,
    };
  }

  /**
   * Returns the value for the entity_types subfield.
   *
   * @return string|void
   *   Returns the value for the entity_types subfield, null otherwise.
   */
  public function getEntityTypes() {
    return $this->entity_types ?: NULL;
  }

  /**
   * Returns the value for the views subfield.
   *
   * @return string|void
   *   Returns the value for the views subfield, null otherwise.
   */
  public function getViews() {
    return $this->views ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints(): array {
    $constraints = parent::getConstraints();
    $constraint_manager = $this->getTypedDataManager()->getValidationConstraintManager();
    $constraints[] = $constraint_manager->create('ComplexData', [
      'entity_types' => [
        'SameEntityType' => [],
      ],
    ]);

    return $constraints;
  }

  /**
   * {@inheritdoc}
   */
  public function setValue($values, $notify = TRUE) {
    // Transform $values['entity_types'] to array.
    if (empty($values['entity_types'])) {
      $values['entity_types'] = [];
    }
    if (!is_array($values['entity_types'])) {
      $values['entity_types'] = [$values['entity_types']];
    }
    parent::setValue($values, $notify);
  }

}
