<?php

declare(strict_types=1);

namespace Drupal\listing_page\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\listing_page\Helper;

/**
 * Defines the 'entity_listing_info_default' field widget.
 */
#[FieldWidget(
  id: 'entity_listing_info_default',
  label: new TranslatableMarkup('Entity listing information'),
  field_types: ['entity_listing_info'],
)]
final class EntityListingInfoDefaultWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Listing page helper.
   *
   * @var \Drupal\listing_page\Helper
   */
  protected $listingPageHelper;

  /**
   * {@inheritdoc}
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, EntityTypeManagerInterface $entity_type_manager, Helper $listing_page_helper) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->entityTypeManager = $entity_type_manager;
    $this->listingPageHelper = $listing_page_helper;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity_type.manager'),
      $container->get('listing_page.helper')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element['#type'] = 'details';
    $element['#open'] = TRUE;
    $element['#group'] = 'advanced';

    $element['views'] = [
      '#type' => 'select',
      '#title' => $this->t('View'),
      '#options' => $this->getAvailableViews(),
      '#empty_option' => t('None'),
      '#empty_value' => '',
      '#default_value' => $items[$delta]->views ?? NULL,
      '#description' => t('Select a view to embed into this entity. Only views with a "Listing page" display are available.'),
    ];

    $element['entity_types'] = [
      '#type' => 'select',
      '#title' => $this->t('Content type'),
      '#options' => $this->getAvailableEntityTypes(),
      '#empty_option' => t('None'),
      '#empty_value' => '',
      '#default_value' => $items[$delta]->entity_types ?? NULL,
      '#description' => t('Select the content entity that this page will be listing. <br>Note mixing different entity types is not supported.'),
      '#multiple' => $this->getFieldSetting('multiple_bundles'),
      '#states' => [
        'visible' => [
          'select[name="' . $this->fieldDefinition->getName() . '[' . $delta . '][views]"]' => ['empty' => FALSE],
        ],
      ],
    ];

    return $element;
  }

  /**
   * Get available listing pages views.
   */
  protected function getAvailableViews() {
    $views = $this->entityTypeManager->getStorage('view')->loadMultiple();

    $options = [];
    foreach ($views as $key => $value) {
      $display = $views[$key]->get('display');
      foreach ($display as $display_id => $value_display) {
        if (($value_display['display_plugin'] == 'listing_page') &&
          (!isset($value_display['display_options']['enabled']) || !empty($value_display['display_options']['enabled']))
          ) {
          $options[$key . ':' . $display_id] = $value->label() . ' : ' . $value_display['display_title'];
        }
      }
    }

    return $options;
  }

  /**
   * Get available listing pages views.
   */
  protected function getAvailableEntityTypes() {
    $entity_types = $this->listingPageHelper->getContentEntityTypes();
    $options = [];
    foreach ($entity_types as $entity_type_id => $entity_type) {
      foreach ($entity_type['bundles'] as $bundle_id => $bundle) {
        $options[$entity_type_id . ':' . $bundle_id] = ($entity_type_id != $bundle_id) ? $entity_type['label'] . ' : ' . $bundle['label'] : $entity_type['label'];
      }
    }
    return $options;
  }

}
