<?php

declare(strict_types=1);

namespace Drupal\listing_page\Twig;

use Drupal\listing_page\Helper;
use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;
use Drupal\Core\Render\Markup;
use Twig\Markup as TwigMarkup;

/**
 * Twig extension.
 */
final class TwigExtension extends AbstractExtension {

  /**
   * Constructs the extension object.
   */
  public function __construct(
    private readonly Helper $listingPageHelper,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function getFunctions(): array {
    $functions[] = new TwigFunction('listing_link', [$this, 'listingEntityLink']);
    return $functions;
  }

  /**
   * Gets a rendered link for a listing page.
   *
   * @param string $entity_type_id
   *   The listed entity type id.
   * @param string $bundle
   *   The listed bundle.
   * @param string $text
   *   The text for the link as a translated string.
   * @param string $rel
   *   The link relationship type, for example: canonical or edit-form. If none
   *   is provided, canonical is assumed, or edit-form if no canonical link
   *   exists.
   * @param array $options
   *   See \Drupal\Core\Routing\UrlGeneratorInterface::generateFromRoute() for
   *   the available options.
   * @param string $langcode
   *   The language code.
   *
   * @return void|array
   *   The link render array.
   */
  public function listingEntityLink(string $entity_type_id, string $bundle, string $text, $rel = 'canonical', array $options = [], ?string $langcode = NULL) {
    if ($listing_entity = $this->listingPageHelper->getListingEntityForBundle($entity_type_id, $bundle)) {
      if (!empty($langcode)) {
        $listing_entity = $listing_entity->getTranslation($langcode);
      }
      // The text has been processed by twig already,
      // convert it to a safe object for the render system.
      if ($text instanceof TwigMarkup) {
        $text = Markup::create($text);
      }

      return [
        '#type' => 'link',
        '#title' => $text,
        '#url' => $listing_entity->toUrl($rel, $options),
      ];
    }
  }

}
