<?php

declare(strict_types=1);

namespace Drupal\listing_page\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\listing_page\Helper;

/**
 * Hook implementations for listing_page (Theming).
 */
class ThemeHooks {

  public function __construct(
    protected readonly Helper $listingHelper,
  ) {}

  /**
   * Implements hook_theme_suggestions_HOOK_alter() for views_view templates.
   */
  #[Hook('theme_suggestions_views_view_alter')]
  public function themeSuggestionsViewsViewAlter(array &$suggestions, array $variables): void {
    if (!in_array('views_view__' . $variables['view']->getDisplay()->getPluginId(), $suggestions)) {
      $suggestions[] = 'views_view__' . $variables['view']->getDisplay()->getPluginId();
    }
    if (!in_array('views_view__' . $variables['view']->id() . '_' . $variables['view']->current_display, $suggestions)) {
      $suggestions[] = 'views_view__' . $variables['view']->id() . '_' . $variables['view']->current_display;
    }
  }

  /**
   * Implements hook_theme_suggestions_HOOK_alter() for views_exposed_form templates.
   */
  #[Hook('theme_suggestions_views_exposed_form_alter')]
  public function themeSuggestionsViewsExposedFormAlter(array &$suggestions, array $variables): void {
    if (!empty($variables['form']['#display_plugin_id'])
    && !in_array('views_exposed_form__' . $variables['form']['#display_plugin_id'], $suggestions)
    ) {
      array_unshift($suggestions, 'views_exposed_form__' . $variables['form']['#display_plugin_id']);
    }
  }

  /**
   * Implements hook_theme_suggestions_HOOK_alter() for form templates.
   */
  #[Hook('theme_suggestions_form_alter')]
  public function themeSuggestionsFormAlter(array &$suggestions, array $variables): void {
    if (!empty($variables['element']['#form_id'] && 'views_exposed_form' == $variables['element']['#form_id'])) {
      if (!empty($variables['element']['#display_plugin_id']) && 'listing_page' == $variables['element']['#display_plugin_id']) {
        $suggestions[] = 'form__views_exposed_form__listing_page';
      }
    }
  }

  /**
   * Implements hook_theme_suggestions_HOOK_alter() for listing entities templates (node, taxonomy_term, etc..)
   */
  #[Hook('theme_suggestions_alter')]
  public function themeSuggestionsAlter(array &$suggestions, array $variables, $hook): void {
    $listing_entity_types = $this->listingHelper->getListingEntitiesDefinitions();
    if (!array_key_exists($hook, $listing_entity_types)) {
      return;
    }
    foreach ($listing_entity_types as $entity_type_id => $bundles) {
      if (!isset($variables['elements']['#' . $entity_type_id])) {
        continue;
      }
      $entity = $variables['elements']['#' . $entity_type_id];
      foreach ($bundles as $bundle => $listing_field_name) {
        if ($entity->bundle() == $bundle) {
          if ($entity->get($listing_field_name)->isEmpty()) {
            $suggestion_prev = $entity_type_id . '__' . $entity->id();
            $suggestion = $entity_type_id . '__' . $entity->bundle() . '__no_views';
            $this->arrayInsertBefore($suggestions, $suggestion, $suggestion_prev);
          }
          else {
            // Vary suggestions on listing views name.
            if ($views_id = $entity->get($listing_field_name)->first()->getViews()) {
              [$views_id, $display_id] = explode(':', $views_id);
              $sanitized_views_id = strtr($views_id, ':', '_');
              $suggestion_prev = $entity_type_id . '__' . $entity->id();
              $suggestion = $entity_type_id . '__' . $entity->bundle() . '__' . $sanitized_views_id;
              $this->arrayInsertBefore($suggestions, $suggestion, $suggestion_prev);
            }
            // Vary suggestions on listed content entity type.
            if ($entity_types = $entity->get($listing_field_name)->first()->getEntityTypes()) {
              $suggestion = '';
              $suggestion_prev = $entity_type_id . '__' . $entity->id();
              foreach ($entity_types as $entity_type) {
                [$target_entity_type_id, $target_bundle] = explode(':', $entity_type);
                $sanitized_entity_type_id = strtr($target_entity_type_id, ':', '_');
                $sanitized_bundle = strtr($target_bundle, ':', '_');
                $suggestion .= '_' . $sanitized_entity_type_id . '_' . $sanitized_bundle;
              }
              $suggestion = $entity_type_id . '__' . $entity->bundle() . '__' . $suggestion;
              $this->arrayInsertBefore($suggestions, $suggestion, $suggestion_prev);
            }
          }
        }
      }
    }

  }

  /**
   * Inserts a new key/value before the key in the array.
   *
   * @param array $haystack
   *   The host associative array.
   * @param mixed $value
   *   The key to insert before.
   * @param string $position
   *   The array to be inserted, should be something like
   *   [$new_key => $new_value].
   *
   * @return bool|array
   *   The new array if the key exists, FALSE otherwise.
   *
   * @see array_insert_after()
   */
  private function arrayInsertBefore(array &$haystack, $value, $position) {
    $key = array_search($position, $haystack);
    if ($key === FALSE) {
      return FALSE;
    }
    if ($key === 0) {
      array_unshift($haystack, $value);
    }
    else {
      $haystack = array_merge(
        array_slice($haystack, 0, $key),
        [$value],
        array_slice($haystack, $key)
          );
    }
    return $haystack;
  }

}
