<?php

/**
 * @file
 * Post update functions for the book module.
 */

use Drupal\Core\Config\Entity\ConfigEntityUpdater;
use Drupal\Core\Entity\Display\EntityViewDisplayInterface;
use Drupal\block\Entity\Block;
use Drupal\user\RoleInterface;

/**
 * Pre-populate the use_top_level_title setting of the book_navigation blocks.
 */
function book_post_update_prepopulate_use_top_level_title_setting(&$sandbox): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'block', function (Block $block) {
    if ($block->getPluginId() === 'book_navigation') {
      $block->getPlugin()->setConfigurationValue('use_top_level_title', FALSE);
      return TRUE;
    }
    return FALSE;
  });
}

/**
 * Update extra book field for entity view displays.
 */
function book_post_update_book_navigation_view_display(?array &$sandbox = NULL): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'entity_view_display', function (EntityViewDisplayInterface $entity_view_display): bool {
    $update = FALSE;
    $components = $entity_view_display->getComponents();
    if ($entity_view_display->getTargetEntityTypeId() === 'node') {
      if (isset($components['book_navigation'])) {
        if ($entity_view_display->getMode() !== 'full' || $entity_view_display->getMode() !== 'default') {
          $updated_component = $entity_view_display->getComponent('book_navigation');
          $updated_component['region'] = 'hidden';
          $entity_view_display->setComponent('book_navigation', $updated_component);
          $update = TRUE;
        }
      }
    }
    return $update;
  });
}

/**
 * Grant 'access book list' permission to roles with 'access content'.
 */
function book_post_update_book_list_permission(?array &$sandbox = NULL): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'user_role', function (RoleInterface $role): bool {
    $update = FALSE;

    if ($role->hasPermission('access content') && !$role->hasPermission('access book list')) {
      $role->grantPermission('access book list');
      $update = TRUE;
    }

    return $update;
  });
}

/**
 * Converts book.settings.allowed_types + child_type to structured format.
 */
function book_post_update_convert_allowed_types_to_structured(array &$sandbox = []): void {
  $config = \Drupal::configFactory()->getEditable('book.settings');

  $old_allowed = $config->get('allowed_types');
  $legacy_child_type = $config->get('child_type');

  // Normalize ANYTHING into a safe structured array.
  $new_allowed = _book_normalize_allowed_types_to_structured($old_allowed, $legacy_child_type);

  // Write if anything changed OR legacy key exists.
  if ($new_allowed !== $old_allowed || $legacy_child_type !== NULL) {
    $config->set('allowed_types', $new_allowed);
    if ($legacy_child_type !== NULL) {
      $config->clear('child_type');
    }
    $types = node_type_get_names();
    foreach (array_keys($types) as $type) {
      $config->clear('allowed_type_' . $type);
      $config->clear('child_type_' . $type);
    }
    $config->clear('block');
    $config->save(TRUE);
  }
}

/**
 * Normalizes old book allowed_types into structured format.
 *
 * @param mixed $old_allowed
 *   The legacy allowed_types value. May be a string, array, or malformed data.
 * @param string|null $legacy_child_type
 *   The legacy global child_type value, if present.
 *
 * @return array
 *   Structured allowed_types entries.
 */
function _book_normalize_allowed_types_to_structured($old_allowed, ?string $legacy_child_type): array {
  $items = [];

  // Normalize input into iterable list.
  if (is_string($old_allowed) && $old_allowed !== '') {
    $items = [$old_allowed];
  }
  elseif (is_array($old_allowed)) {
    $items = $old_allowed;
  }
  else {
    return [];
  }

  $normalized = [];

  foreach ($items as $item) {
    $content_type = NULL;
    $child_type = NULL;

    // Flat string.
    if (is_string($item) && $item !== '') {
      $content_type = $item;
    }
    // Structured or broken array.
    elseif (is_array($item)) {
      if (isset($item['content_type'])) {
        if (is_string($item['content_type']) && $item['content_type'] !== '') {
          $content_type = $item['content_type'];
        }
        elseif (
          is_array($item['content_type']) &&
          isset($item['content_type']['content_type']) &&
          is_string($item['content_type']['content_type'])
        ) {
          $content_type = $item['content_type']['content_type'];
        }
      }

      if (isset($item['child_type']) && is_string($item['child_type']) && $item['child_type'] !== '') {
        $child_type = $item['child_type'];
      }
    }

    if ($content_type) {
      $normalized[] = [
        'content_type' => $content_type,
        'child_type' => $legacy_child_type ?: $child_type ?: $content_type,
      ];
    }
  }

  return $normalized;
}

/**
 * Pre-populate the show_top_item setting of the book_navigation blocks.
 */
function book_post_update_prepopulate_show_top_item_setting(&$sandbox): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'block', function (Block $block) {
    if ($block->getPluginId() === 'book_navigation') {
      $block->getPlugin()->setConfigurationValue('show_top_item', FALSE);
      return TRUE;
    }
    return FALSE;
  });
}

/**
 * Add the default label truncation setting.
 */
function book_post_update_add_default_label_truncate_settings(): ?string {
  $config = \Drupal::configFactory()->getEditable('book.settings');
  if ($config->get('truncate_label') === NULL) {
    // Specify the default setting if not specified.
    $config
      ->set('truncate_label', TRUE)
      ->save(TRUE);
    return 'Updated the default truncate setting.';
  }
  return NULL;
}

/**
 * Remove 'add any content to books' permission.
 */
function book_post_update_remove_add_any_content_books_permission(?array &$sandbox = NULL): void {
  \Drupal::classResolver(ConfigEntityUpdater::class)->update($sandbox, 'user_role', function (RoleInterface $role): bool {
    $update = FALSE;
    $permission = 'add any content to books';

    if ($role->hasPermission('add any content to books')) {
      $role->revokePermission($permission);
      $update = TRUE;
    }

    return $update;
  });
}

/**
 * Set allowed child_type to parent if none previously configured.
 */
function book_post_update_allowed_child_type_default6(array &$sandbox = []): void {
  $config_factory = \Drupal::configFactory();
  $config = $config_factory->getEditable('book.settings');
  $updated = FALSE;

  $allowed_types = $config->get('allowed_types');
  if ($allowed_types === NULL) {
    return;
  }
  $normalized = [];

  // If stored as a single string for some reason.
  if (is_string($allowed_types) && $allowed_types !== '') {
    $allowed_types = [$allowed_types];
  }

  if (is_array($allowed_types)) {
    foreach ($allowed_types as $item) {
      $content_type = NULL;
      $child_type = NULL;

      // Old style: ['page', 'article'].
      if (is_string($item) && $item !== '') {
        $content_type = $item;
        $child_type = $item;
        $updated = TRUE;
      }
      // New-ish / broken style.
      elseif (is_array($item)) {
        // The important part: flatten nested content_type structures.
        if (isset($item['content_type'])) {
          if (is_string($item['content_type'])) {
            $content_type = $item['content_type'];
          }
          elseif (is_array($item['content_type'])) {
            // Handles: ['content_type' => ['content_type' => 'page']].
            if (isset($item['content_type']['content_type']) && is_string($item['content_type']['content_type'])) {
              $content_type = $item['content_type']['content_type'];
              $updated = TRUE;
            }
          }
        }

        if ($content_type) {
          // Child type: string or default to content_type.
          if (isset($item['child_type']) && is_string($item['child_type']) && $item['child_type'] !== '') {
            $child_type = $item['child_type'];
          }
          else {
            $child_type = $content_type;
            $updated = TRUE;
          }
        }
      }

      // Only accept valid strings; drop anything else.
      if (is_string($content_type) && $content_type !== '' && is_string($child_type) && $child_type !== '') {
        $normalized[] = [
          'content_type' => $content_type,
          'child_type' => $child_type,
        ];
      }
      else {
        $updated = TRUE;
      }
    }
  }

  if ($updated) {
    $config->set('allowed_types', $normalized)->save(TRUE);
  }
}
