<?php

/**
 * @file
 * File to add menu markdown tokens.
 */

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\system\Entity\Menu;

/**
 * Implements hook_token_info().
 */
function llms_txt_token_info(): array {
  $types = [];
  $types['llms_txt_markdown_menu'] = [
    'name' => t('llms.txt: Markdown menu'),
    'description' => t('Render menu items in markdown for llms.txt'),
  ];

  /** @var \Drupal\system\MenuInterface[] $menus */
  $menus = Drupal::entityTypeManager()->getStorage('menu')->loadMultiple();

  $tokens = [];
  foreach ($menus as $menu) {
    $tokens[$menu->id()] = [
      'name' => $menu->label(),
      'description' => $menu->getDescription(),
    ];
  }

  return [
    'types' => $types,
    'tokens' => ['llms_txt_markdown_menu' => $tokens],
  ];
}

/**
 * Implements hook_tokens().
 */
function llms_txt_tokens(string $type, array $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
  $replacements = [];

  if ($type === 'llms_txt_markdown_menu') {
    $manipulators = [
      ['callable' => 'menu.default_tree_manipulators:checkAccess'],
      ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
      ['callable' => 'menu.default_tree_manipulators:flatten'],
    ];

    $parameters = new MenuTreeParameters();
    // Make this configurable if necessary.
    $parameters->setMaxDepth(3);
    $parameters->onlyEnabledLinks();

    $menu_tree = \Drupal::menuTree();

    /** @var \Drupal\system\MenuInterface[] $menus */
    $menus = Menu::loadMultiple(array_keys($tokens));
    foreach ($menus as $menu_name => $menu) {
      $bubbleable_metadata->addCacheableDependency($menu);
      $tree = $menu_tree->load($menu_name, $parameters);
      $tree = $menu_tree->transform($tree, $manipulators);

      $lines = [];
      foreach ($tree as $item) {
        if ($item->access === NULL) {
          continue;
        }

        $bubbleable_metadata->addCacheableDependency($item->access);
        if (!$item->access->isAllowed()) {
          continue;
        }

        $bubbleable_metadata->addCacheableDependency($item->link);

        $prefix = str_repeat('  ', $item->depth - 1);
        $title = $item->link->getTitle();
        $url = $item->link->getUrlObject()->setAbsolute()->toString();
        $details = (string) $item->link->getDescription();

        $line = sprintf('%s- [%s](%s)', $prefix, $title, $url);
        if ($details !== '') {
          $line .= ': ' . $details;
        }
        $lines[] = $line;
      }

      // @phpstan-ignore offsetAccess.invalidOffset
      $replacements[$tokens[$menu_name]] = implode(PHP_EOL, $lines);
    }
  }

  return $replacements;
}
