<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */
namespace Drupal\llms_txt\Entity;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\RefinableCacheableDependencyInterface;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Entity access handler for llms.txt section entity type.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class LlmsTxtSectionAccessHandler extends EntityAccessControlHandler {

  /**
   * {@inheritdoc}
   */
  protected function checkAccess(EntityInterface $entity, $operation, AccountInterface $account) {
    assert($entity instanceof LlmsTxtSection);
    $cacheability = new CacheableMetadata();
    if ($operation === 'view') {
      $cacheability->addCacheableDependency($entity);
      // All published llms.txt section entities are intended to be publicly
      // accessible.
      if ($entity->isPublished()) {
        return AccessResult::allowed()->addCacheableDependency($cacheability);
      }
    }
    $result = parent::checkAccess($entity, $operation, $account);
    if ($result instanceof RefinableCacheableDependencyInterface) {
      $result->addCacheableDependency($cacheability);
    }
    else {
      $result = AccessResult::neutral()->addCacheableDependency($cacheability)->orIf($result);
    }

    return $result;
  }

}
