<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */
namespace Drupal\llms_txt\HttpKernel;

use Drupal\Core\PathProcessor\InboundPathProcessorInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Ensures /llms.txt is only available on the default path.
 *
 * This processor prevents the /llms.txt endpoint from being accessible through
 * language-prefixed paths (e.g., /en/llms.txt, /de/llms.txt) by redirecting
 * such requests to a non-existent path, which triggers a 404 response.
 *
 * This implementation is necessary for technical endpoints like llms.txt that
 * should remain consistent across all language contexts and not be subject
 * to Drupal's automatic language negotiation mechanisms.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class LlmsTxtOnlyOnDefaultPathProcessor implements InboundPathProcessorInterface {

  private const LLMS_TXT_PATH = '/llms.txt';
  private const BLOCKED_PATH_SUFFIX = '-only-served-on-default-language';

  /**
   * {@inheritdoc}
   */
  public function processInbound($path, Request $request): string {
    if ($path === self::LLMS_TXT_PATH) {
      return $path;
    }

    if (str_ends_with($path, self::LLMS_TXT_PATH)) {
      // Return non-existent path to trigger 404 in RouterListener.
      // @see \Symfony\Component\HttpKernel\EventListener\RouterListener::onKernelRequest()
      return self::LLMS_TXT_PATH . self::BLOCKED_PATH_SUFFIX;
    }

    return $path;
  }

}
