<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA.
 */

namespace Drupal\llms_txt\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Defines the llms.txt section entity class.
 *
 * @ContentEntityType(
 *   id = "llms_txt_section",
 *   label = @Translation("llms.txt section"),
 *   label_collection = @Translation("llms.txt sections"),
 *   label_singular = @Translation("llms.txt section"),
 *   label_plural = @Translation("llms.txt sections"),
 *   label_count = @PluralTranslation(
 *     singular = "@count llms.txt sections",
 *     plural = "@count llms.txt sections",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\llms_txt\Form\LlmsTxtSectionOverviewForm",
 *     "access" = "Drupal\llms_txt\Entity\LlmsTxtSectionAccessHandler",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\llms_txt\Form\LlmsTxtSectionForm",
 *       "edit" = "Drupal\llms_txt\Form\LlmsTxtSectionForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "llms_txt_section",
 *   admin_permission = "administer llms.txt configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "title",
 *     "uuid" = "uuid",
 *     "weight" = "weight",
 *     "published" = "status",
 *   },
 *   links = {
 *     "collection" = "/admin/content/llms-txt/sections",
 *     "add-form" = "/admin/content/llms-txt/add-section",
 *     "canonical" = "/admin/content/llms-txt/sections/{llms_txt_section}",
 *     "edit-form" = "/admin/content/llms-txt/sections/{llms_txt_section}/edit",
 *     "delete-form" = "/admin/content/llms-txt/sections/{llms_txt_section}/delete",
 *     "delete-multiple-form" = "/admin/content/llms-txt/sections/delete-multiple",
 *   },
 * )
 *
 * @internal This class is not part of the module's public programming API.
 */
final class LlmsTxtSection extends ContentEntityBase implements LlmsTxtSectionInterface, EntityPublishedInterface {

  use EntityPublishedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['title'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Title'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['content'] = BaseFieldDefinition::create('text_long')
      ->setLabel(new TranslatableMarkup('Content'))
      ->setDescription(new TranslatableMarkup('Enter Markdown formatted text only. Avoid using HTML tags as this content will appear in <a href="/llms.txt" target="_blank">llms.txt file</a>, which adheres to the <a href="https://llmstxt.org/" target="_blank" rel="noopener">llms.txt specification</a> for AI-readable content.'))
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'label' => 'above',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(new TranslatableMarkup('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['weight'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Weight'))
      ->setDescription(new TranslatableMarkup('The weight of this term in relation to other terms.'))
      ->setDefaultValue(0);

    return $fields;
  }

}
