<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA.
 */

namespace Drupal\llms_txt\Form;

use Drupal\Core\Entity\DraggableListBuilderTrait;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormInterface;
use Drupal\llms_txt\Entity\LlmsTxtSectionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Admin form for managing llms.txt section entities.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class LlmsTxtSectionOverviewForm extends EntityListBuilder implements FormInterface {

  use DraggableListBuilderTrait {
    buildHeader as traitBuildHeader;
    buildRow as traitBuildRow;
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage) {
    parent::__construct($entity_type, $storage);

    // Check if the entity type supports weighting.
    if ($this->entityType->hasKey('weight')) {
      $this->weightKey = $this->entityType->getKey('weight');
    }
    $this->limit = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): self {
    $instance = parent::createInstance($container, $entity_type);

    $instance->formBuilder = $container->get('form_builder');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return $this->entityType->id() . '_overview_form';
  }

  /**
   * {@inheritdoc}
   */
  public function load() {
    $query = $this->storage->getQuery()->accessCheck(FALSE);
    $query->sort('weight');
    $entity_ids = $query->execute();
    if ($entity_ids === []) {
      return [];
    }
    return $this->storage->loadMultiple($entity_ids);
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header = [];
    $header['label'] = $this->t('Section title');
    $header['description'] = $this->t('Status');
    return $header + $this->traitBuildHeader() + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    assert($entity instanceof LlmsTxtSectionInterface);
    $row = [];
    $row['label'] = $entity->label();
    // @phpstan-ignore offsetAccess.nonOffsetAccessible
    $row['status']['data'] = $entity->get('status')->view(['label' => 'hidden']);
    return $row + $this->traitBuildRow($entity) + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function getWeight(EntityInterface $entity): int {
    assert($entity instanceof LlmsTxtSectionInterface);
    return (int) $entity->get($this->weightKey)->value ?: 0;
  }

  /**
   * {@inheritdoc}
   */
  protected function setWeight(EntityInterface $entity, int|float $weight): EntityInterface {
    assert($entity instanceof LlmsTxtSectionInterface);
    $entity->set($this->weightKey, $weight);
    return $entity;
  }

}
