<?php

/**
 * @file
 * File to add menu markdown tokens.
 */

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 */

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\system\Entity\Menu;

/**
 * Implements hook_token_info().
 */
function llms_txt_token_info(): array {
  $info = [
    'types' => [],
    'tokens' => [],
  ];
  $info['types']['llms_txt_markdown_menu'] = [
    'name' => t('llms.txt: Markdown menu'),
    'description' => t('Tokens that render menu items as Markdown-formatted lists for inclusion in llms.txt files.'),
  ];
  if (\Drupal::moduleHandler()->moduleExists('markdownify_views')) {
    $info['types']['llms_txt_views'] = [
      'name' => t('llms.txt: Views'),
      'description' => t('Tokens that render Views output as Markdown content for llms.txt files. Views must be tagged with "llms_txt_section" and configured to produce Markdown-compatible output (e.g., urls should use absolute URLs suffixed with .md for markdown versions). Contextual arguments are not supported.'),
    ];
    /** @var \Drupal\views\ViewEntityInterface[] $views */
    $views = \Drupal::service('entity_type.manager')
      ->getStorage('view')
      ->loadByProperties(['tag' => 'llms_txt_section']);
    foreach ($views as $view) {
      foreach ($view->get('display') as $display) {
        $info['tokens']['llms_txt_views'][$view->id() . ':' . $display['id']] = [
          'name' => sprintf('%s (%s)', $view->label(), $display['display_title']),
          'description' => t('Renders the "@display" display of the "@view" view as Markdown content.', [
            '@display' => $display['display_title'],
            '@view' => $view->label(),
          ]),
        ];
      }
    }
  }

  /** @var \Drupal\system\MenuInterface[] $menus */
  $menus = Drupal::entityTypeManager()->getStorage('menu')->loadMultiple();
  foreach ($menus as $menu) {
    $info['tokens']['llms_txt_markdown_menu'][$menu->id()] = [
      'name' => $menu->label(),
      'description' => $menu->getDescription() ?: t('Renders the "@menu" menu as a Markdown-formatted list.', [
        '@menu' => $menu->label(),
      ]),
    ];
  }

  return $info;
}

/**
 * Helper function to recursively traverse menu tree in depth-first order.
 *
 * @param \Drupal\Core\Menu\MenuLinkTreeElement[] $tree
 *   The menu tree to traverse.
 * @param array $lines
 *   Array to collect formatted lines (passed by reference).
 * @param \Drupal\Core\Render\BubbleableMetadata $bubbleable_metadata
 *   Bubbleable metadata to collect cache info.
 * @param int $depth
 *   Current depth level (1-based).
 */
function _llms_txt_traverse_menu_tree(array $tree, array &$lines, BubbleableMetadata $bubbleable_metadata, int $depth = 1): void {
  assert($depth >= 1);
  foreach ($tree as $item) {
    // Check access.
    if ($item->access === NULL) {
      continue;
    }

    $bubbleable_metadata->addCacheableDependency($item->access);
    if (!$item->access->isAllowed()) {
      continue;
    }

    $bubbleable_metadata->addCacheableDependency($item->link);

    // Format the current item.
    $prefix = str_repeat('  ', $depth - 1);
    $title = $item->link->getTitle();
    $url = $item->link->getUrlObject()->setAbsolute()->toString();
    $details = (string) $item->link->getDescription();

    $line = sprintf('%s- [%s](%s)', $prefix, $title, $url);
    if ($details !== '') {
      $line .= ': ' . $details;
    }
    $lines[] = $line;

    // Recursively process children immediately after parent (depth-first).
    if ($item->hasChildren && !empty($item->subtree)) {
      _llms_txt_traverse_menu_tree($item->subtree, $lines, $bubbleable_metadata, $depth + 1);
    }
  }
}

/**
 * Implements hook_tokens().
 */
function llms_txt_tokens(string $type, array $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
  $replacements = [];

  if ($type === 'llms_txt_markdown_menu') {
    $manipulators = [
      ['callable' => 'menu.default_tree_manipulators:checkAccess'],
      ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
    ];

    $parameters = new MenuTreeParameters();
    // Make this configurable if necessary.
    $parameters->setMaxDepth(3);
    $parameters->onlyEnabledLinks();

    $menu_tree = \Drupal::menuTree();

    /** @var \Drupal\system\MenuInterface[] $menus */
    $menus = Menu::loadMultiple(array_keys($tokens));
    foreach ($menus as $menu_name => $menu) {
      $bubbleable_metadata->addCacheableDependency($menu);
      $tree = $menu_tree->load($menu_name, $parameters);
      $tree = $menu_tree->transform($tree, $manipulators);

      $lines = [];
      // Recursive depth-first traversal to maintain parent-child adjacency.
      _llms_txt_traverse_menu_tree($tree, $lines, $bubbleable_metadata);

      // @phpstan-ignore offsetAccess.invalidOffset
      $replacements[$tokens[$menu_name]] = implode(PHP_EOL, $lines);
    }
  }
  elseif ($type === 'llms_txt_views' && \Drupal::moduleHandler()->moduleExists('markdownify_views')) {
    foreach ($tokens as $name => $original) {
      [$view_id, $display_id] = explode(':', $name, 2);
      if (!empty($view_id) && !empty($display_id)) {
        $build = views_embed_view($view_id, $display_id);
        if (is_array($build)) {
          $html = \Drupal::service('renderer')->renderInIsolation($build);
          $bubbleable_metadata->addCacheableDependency(CacheableMetadata::createFromRenderArray($build));
          $markdown = \Drupal::service('markdownify.html_converter')->convert((string) $html);
          $replacements[$original] = $markdown;
        }
      }
    }
  }

  return $replacements;
}
