<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA.
 */

namespace Drupal\llms_txt\Controller;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\AutowireTrait;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Utility\Token;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller that serves the /llms.txt file.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class LlmsTxtController implements ContainerInjectionInterface {

  use AutowireTrait;

  /**
   * Creates a new instance.
   */
  public function __construct(
    private readonly Token $token,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly RendererInterface $renderer,
  ) {}

  /**
   * Serves the llms.txt file content.
   *
   * @return \Drupal\Core\Cache\CacheableResponse
   *   The llms.txt file content.
   */
  public function __invoke(): CacheableResponse {
    $cacheability = new CacheableMetadata();
    $config = $this->configFactory->get('llms_txt.settings');
    $cacheability->addCacheableDependency($config);

    $token_metadata = new BubbleableMetadata();
    $content = [
      $this->token->replace($config->get('content'), [], [], $token_metadata),
    ];
    $cacheability->addCacheableDependency($token_metadata);

    $sections_entity_def = $this->entityTypeManager->getDefinition('llms_txt_section');
    $cacheability->addCacheTags($sections_entity_def->getListCacheTags());
    $cacheability->addCacheContexts($sections_entity_def->getListCacheContexts());

    $sections_storage = $this->entityTypeManager->getStorage('llms_txt_section');
    $entity_ids = $sections_storage->getQuery()->accessCheck()->condition('status', 1)->sort('weight')->execute();

    /** @var \Drupal\llms_txt\Entity\LlmsTxtSectionInterface $entity */
    foreach ($sections_storage->loadMultiple($entity_ids) as $entity) {
      $entity_access = $entity->access('view', return_as_object: TRUE);
      $cacheability->addCacheableDependency($entity_access);
      if ($entity_access->isAllowed()) {
        $cacheability->addCacheableDependency($entity);
        $content_field_access = $entity->get('content')->access('view', return_as_object: TRUE);
        $cacheability->addCacheableDependency($content_field_access);
        // Only include the entity in the result if access to the content field
        // is granted.
        if ($content_field_access->isAllowed()) {
          // Render field value without the default field wrapper markup.
          $build = [
            '#type' => 'processed_text',
            '#text' => $entity->get('content')->value,
            // @phpstan-ignore property.notFound
            '#format' => $entity->get('content')->format,
          ];
          $content[] = sprintf("## %s\n%s\n", $entity->label(), $this->renderer->renderInIsolation($build));
          $cacheability->addCacheableDependency(CacheableMetadata::createFromRenderArray($build));
        }
      }
    }

    $response = new CacheableResponse(
      implode("\n", $content),
      Response::HTTP_OK,
      ['Content-Type' => 'text/markdown; charset=utf-8']
    );

    $response->addCacheableDependency($cacheability);

    return $response;
  }

}
