<?php

declare(strict_types = 1);

/**
 * Copyright (C) 2025 PRONOVIX GROUP.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301,
 * USA.
 */

namespace Drupal\llms_txt\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;

/**
 * Configure the content of llms.txt.
 *
 * @internal This class is not part of the module's public programming API.
 */
final class LlmsTxtConfigForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'llms_txt_llms_txt_config';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['llms_txt.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('llms_txt.settings');
    $content = $config->get('content');

    $form['intro'] = [
      '#markup' => $this->t('<a href="https://llmstxt.org/" target="_blank" rel="noopener">llms.txt</a> is a public Markdown file designed to be both human- and machine-readable. It provides a standardized way for Large Language Models (LLMs) and other automated tools to understand your site. The format is easy to read and write, and can be reliably parsed by software.'),
    ];

    $form['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Content of the llms.txt'),
      '#default_value' => $content,
      '#required' => TRUE,
      '#rows' => 8,
      '#wysiwyg' => FALSE,
      '#attributes' => [
        'autofocus' => TRUE,
      ],
    ];

    $form['tokens'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => [],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $content_submit = $form_state->getValue('content');

    $this->config('llms_txt.settings')
      ->set('content', $content_submit)
      ->save();

    $this->messenger()->addStatus($this->t('The content of llms.txt has been saved, you can preview it <a href="@url" target="_blank">here.</a>', [
      '@url' => Url::fromRoute('llms_txt.llms_txt', [], ['absolute' => TRUE])->toString(),
    ]));

    parent::submitForm($form, $form_state);
  }

}
