<?php

namespace Drupal\llms_txt_ai\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\llms_txt_ai\Service\MenuExtractorService;
use Drupal\llms_txt_ai\Service\MetaDescriptionService;
use Drupal\llms_txt_ai\Service\DescriptionStorageService;
use Drupal\llms_txt_ai\Service\ChangeDetectionService;
use Drupal\llms_txt_ai\Service\ReformulatedStorageService;
use Drupal\llms_txt_ai\Service\AiReformulationService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to manage page descriptions.
 */
class DescriptionsManagementForm extends FormBase {

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The meta description service.
   *
   * @var \Drupal\llms_txt_ai\Service\MetaDescriptionService
   */
  protected $metaDescription;

  /**
   * The description storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\DescriptionStorageService
   */
  protected $descriptionStorage;

  /**
   * The change detection service.
   *
   * @var \Drupal\llms_txt_ai\Service\ChangeDetectionService
   */
  protected $changeDetection;

  /**
   * The reformulated storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\ReformulatedStorageService
   */
  protected $reformulatedStorage;

  /**
   * The AI reformulation service.
   *
   * @var \Drupal\llms_txt_ai\Service\AiReformulationService
   */
  protected $aiReformulation;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->menuExtractor = $container->get('llms_txt_ai.menu_extractor');
    $instance->metaDescription = $container->get('llms_txt_ai.meta_description');
    $instance->descriptionStorage = $container->get('llms_txt_ai.description_storage');
    $instance->changeDetection = $container->get('llms_txt_ai.change_detection');
    $instance->reformulatedStorage = $container->get('llms_txt_ai.reformulated_storage');
    $instance->aiReformulation = $container->get('llms_txt_ai.ai_reformulation');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'llms_txt_ai_descriptions_management_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('llms_txt_ai.settings');
    $menus = $config->get('menus') ?? [];

    if (empty($menus)) {
      $form['empty'] = [
        '#markup' => $this->t('No menus configured. Please <a href="@url">configure settings</a> first.', [
          '@url' => '/admin/config/content/llms-txt-ai',
        ]),
      ];
      return $form;
    }

    // Extract pages grouped by menu with hierarchy.
    $menus_data = $this->menuExtractor->extractPagesGroupedByMenu($menus);

    if (empty($menus_data)) {
      $form['empty'] = [
        '#markup' => $this->t('No pages found in selected menus.'),
      ];
      return $form;
    }

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Review and manage descriptions for pages extracted from your menus. Manual descriptions will override automatic meta descriptions.') . '</p>',
    ];

    // Check if homepage should be included.
    $include_homepage = $config->get('include_homepage') ?? TRUE;

    if ($include_homepage) {
      // Try to detect homepage automatically.
      $homepage = $this->menuExtractor->getHomepage();

      // Check if homepage is already in one of the menus.
      $is_duplicate = FALSE;
      if ($homepage) {
        $nid = $homepage['nid'];
        foreach ($menus_data as $menu_data) {
          foreach ($menu_data['pages'] as $page) {
            if ($page['nid'] === $nid) {
              $is_duplicate = TRUE;
              break 2;
            }
          }
        }
      }

      // Always display homepage section if not in a menu.
      if (!$is_duplicate) {
        // Section title.
        $form['homepage_title'] = [
          '#type' => 'markup',
          '#markup' => '<h3 style="color: #4CAF50;">🏠 ' . $this->t('Homepage') . '</h3>',
        ];

        // Table for homepage.
        $form['homepage_table'] = [
          '#type' => 'table',
          '#header' => [
            $this->t('Page'),
            $this->t('URL'),
            $this->t('Meta Description (auto)'),
            $this->t('Manual Description'),
            $this->t('Final Description'),
            $this->t('Status'),
            $this->t('Last Reformulated'),
          ],
        ];

        // If homepage detected, use real data. Otherwise, use manual entry.
        if ($homepage) {
          $nid = $homepage['nid'];
          $title = $homepage['title'];
          $url = $homepage['url'];
          $meta = $this->metaDescription->getMetaDescription($nid);
          $status = $this->changeDetection->getPageStatus($homepage);
        }
        else {
          // Homepage not detected - use special NID for manual entry.
          $nid = 'homepage_manual';
          $title = $this->t('Homepage');
          $url = '/';
          $meta = '';
          $status = 'manual';
        }

        // Get manual description and reformulated data.
        $manual = $this->descriptionStorage->get($nid);
        $reformulated_data = $this->reformulatedStorage->get($nid);

        // Build row.
        $form['homepage_table'][0]['page'] = [
          '#markup' => '<strong>' . $title . '</strong> <span style="background: #4CAF50; color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px;">🏠 Homepage</span>',
        ];

        $form['homepage_table'][0]['url'] = [
          '#markup' => '<code>' . $url . '</code>',
        ];

        $form['homepage_table'][0]['meta'] = [
          '#markup' => !empty($meta) ? $meta : '<em>' . $this->t('No meta description') . '</em>',
        ];

        // Manual description field with AI reformulation button.
        $form['homepage_table'][0]['manual'] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['manual-description-container']],
        ];

        $form['homepage_table'][0]['manual']['textarea'] = [
          '#type' => 'textarea',
          '#default_value' => $manual ?? '',
          '#rows' => 2,
          '#attributes' => [
            'placeholder' => $homepage ? $this->t('Leave empty to use meta description or AI reformulation') : $this->t('Enter homepage description manually'),
            'data-nid' => $nid,
          ],
        ];

        // AI reformulation button.
        $form['homepage_table'][0]['manual']['ai_button'] = [
          '#type' => 'button',
          '#value' => $this->t('✨ AI'),
          '#name' => 'ai_button_homepage_' . $nid,
          '#ajax' => [
            'callback' => '::reformulateDescriptionAjax',
            'event' => 'click',
            'progress' => [
              'type' => 'throbber',
              'message' => $this->t('Generating...'),
            ],
          ],
          '#attributes' => [
            'class' => ['button', 'button--small'],
            'data-nid' => $nid,
            'data-title' => $title,
            'title' => $this->t('Reformulate with AI'),
            'style' => 'margin-top: 5px;',
          ],
        ];

        // Get final description that will be used.
        $final_description = $this->getFinalDescription($nid, $title);
        $form['homepage_table'][0]['final'] = [
          '#markup' => $this->formatFinalDescription($final_description),
        ];

        // Status badge.
        $form['homepage_table'][0]['status'] = [
          '#markup' => $this->getStatusBadge($status),
        ];

        // Last reformulated.
        $last_reformulated = '';
        if (!empty($reformulated_data['timestamp'])) {
          $last_reformulated = date('Y-m-d H:i', $reformulated_data['timestamp']);
          if (!empty($reformulated_data['ai_provider'])) {
            $last_reformulated .= '<br><small>' . $reformulated_data['ai_provider'] . '</small>';
          }
        }
        $form['homepage_table'][0]['last_reformulated'] = [
          '#markup' => $last_reformulated ?: '<em>' . $this->t('Never') . '</em>',
        ];

        // Store NID for submission.
        $form['homepage_table'][0]['nid'] = [
          '#type' => 'hidden',
          '#value' => $nid,
        ];

        // Spacer.
        $form['homepage_spacer'] = [
          '#markup' => '<div style="margin-bottom: 30px;"></div>',
        ];
      }
      elseif ($is_duplicate) {
        // Homepage is in a menu - show info message.
        $form['homepage_info'] = [
          '#type' => 'markup',
          '#markup' => '<div class="messages messages--status" style="margin-bottom: 20px;">' .
            '<strong>ℹ️ ' . $this->t('Homepage Information') . '</strong><br>' .
            $this->t('Your homepage "@title" is already included in one of the selected menus below. You can edit it there (look for the 🏠 Homepage badge).', [
              '@title' => $homepage['title'],
            ]) .
            '</div>',
        ];
      }
    }

    // Build a separate table for each menu.
    foreach ($menus_data as $menu_data) {
      $menu_id = $menu_data['menu_id'];
      $menu_name = $menu_data['menu_name'];
      $pages = $menu_data['pages'];

      // Menu section title.
      $form['menu_' . $menu_id . '_title'] = [
        '#type' => 'markup',
        '#markup' => '<h3>' . $this->t('@menu_name', ['@menu_name' => $menu_name]) . '</h3>',
      ];

      // Table for this menu.
      $form['menu_' . $menu_id] = [
        '#type' => 'table',
        '#header' => [
          $this->t('Page'),
          $this->t('URL'),
          $this->t('Meta Description (auto)'),
          $this->t('Manual Description'),
          $this->t('Final Description'),
          $this->t('Status'),
          $this->t('Last Reformulated'),
        ],
        '#empty' => $this->t('No pages found.'),
      ];

      // Local index for this menu's table (Drupal reindexes each table starting at 0).
      $local_index = 0;
      foreach ($pages as $page) {
        $nid = $page['nid'];
        $depth = $page['depth'] ?? 0;
        $meta = $this->metaDescription->getMetaDescription($nid);
        $manual = $this->descriptionStorage->get($nid);
        $status = $this->changeDetection->getPageStatus($page);
        $reformulated_data = $this->reformulatedStorage->get($nid);

        // Check if this page is the homepage.
        $is_homepage = FALSE;
        if ($include_homepage && isset($homepage) && $homepage['nid'] === $nid) {
          $is_homepage = TRUE;
        }

        // Build indentation prefix (— for each level).
        $indent_prefix = str_repeat('— ', $depth);
        $indent_style = 'padding-left: ' . ($depth * 20) . 'px;';

        // Build title with homepage badge if applicable.
        $title_markup = '<div style="' . $indent_style . '">' . $indent_prefix . $page['title'];
        if ($is_homepage) {
          $title_markup .= ' <span style="background: #4CAF50; color: white; padding: 2px 8px; border-radius: 3px; font-size: 11px;">🏠 Homepage</span>';
        }
        $title_markup .= '</div>';

        $form['menu_' . $menu_id][$local_index]['title'] = [
          '#markup' => $title_markup,
        ];

        $form['menu_' . $menu_id][$local_index]['url'] = [
          '#markup' => $page['url'],
        ];

        $form['menu_' . $menu_id][$local_index]['meta'] = [
          '#markup' => !empty($meta) ? $meta : '<em>' . $this->t('None') . '</em>',
        ];

        // Manual description field with AI reformulation button.
        // KISS: Use data-nid attribute for simple and reliable targeting.
        $form['menu_' . $menu_id][$local_index]['manual'] = [
          '#type' => 'container',
          '#attributes' => ['class' => ['manual-description-container']],
        ];

        $form['menu_' . $menu_id][$local_index]['manual']['textarea'] = [
          '#type' => 'textarea',
          '#default_value' => $manual ?? '',
          '#rows' => 2,
          '#attributes' => [
            'placeholder' => $this->t('Leave empty to use meta description or AI reformulation'),
            'data-nid' => $nid,
          ],
        ];

        // AI reformulation button.
        // IMPORTANT: Use unique name for each button to avoid conflicts.
        $form['menu_' . $menu_id][$local_index]['manual']['ai_button'] = [
          '#type' => 'button',
          '#value' => $this->t('✨ AI'),
          '#name' => 'ai_button_' . $nid,
          '#ajax' => [
            'callback' => '::reformulateDescriptionAjax',
            'event' => 'click',
            'progress' => [
              'type' => 'throbber',
              'message' => $this->t('Generating...'),
            ],
          ],
          '#attributes' => [
            'class' => ['button', 'button--small'],
            'data-nid' => $nid,
            'data-title' => $page['title'],
            'title' => $this->t('Reformulate with AI'),
            'style' => 'margin-top: 5px;',
          ],
        ];

        // Get final description that will be used.
        $final_description = $this->getFinalDescription($nid, $page['title']);
        $form['menu_' . $menu_id][$local_index]['final'] = [
          '#markup' => $this->formatFinalDescription($final_description),
        ];

        $form['menu_' . $menu_id][$local_index]['status'] = [
          '#markup' => $this->getStatusBadge($status),
        ];

        $last_reformulated = '';
        if (!empty($reformulated_data['timestamp'])) {
          $last_reformulated = date('Y-m-d H:i', $reformulated_data['timestamp']);
          if (!empty($reformulated_data['ai_provider'])) {
            $last_reformulated .= '<br><small>' . $reformulated_data['ai_provider'] . '</small>';
          }
        }
        $form['menu_' . $menu_id][$local_index]['last_reformulated'] = [
          '#markup' => $last_reformulated ?: '<em>' . $this->t('Never') . '</em>',
        ];

        // Store NID for submission.
        $form['menu_' . $menu_id][$local_index]['nid'] = [
          '#type' => 'hidden',
          '#value' => $nid,
        ];

        $local_index++;
      }
    }

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save descriptions'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('llms_txt_ai.settings');
    $menus = $config->get('menus') ?? [];

    // Process homepage table if it exists.
    $homepage_data = $form_state->getValue('homepage_table');
    if (!empty($homepage_data) && isset($homepage_data[0])) {
      $homepage_row = $homepage_data[0];
      $nid = $homepage_row['nid'];
      // The manual field is a container with a textarea child.
      $manual_description = trim($homepage_row['manual']['textarea'] ?? '');

      if (!empty($manual_description)) {
        // Save manual description.
        $this->descriptionStorage->set($nid, $manual_description);
      }
      else {
        // Remove manual description if empty.
        $this->descriptionStorage->delete($nid);
      }
    }

    // Process each menu's table.
    foreach ($menus as $menu_id) {
      $menu_pages = $form_state->getValue('menu_' . $menu_id);

      if (!empty($menu_pages)) {
        foreach ($menu_pages as $page_data) {
          $nid = $page_data['nid'];
          // The manual field is now a container with a textarea child.
          $manual_description = trim($page_data['manual']['textarea'] ?? '');

          if (!empty($manual_description)) {
            // Save manual description.
            $this->descriptionStorage->set($nid, $manual_description);
          }
          else {
            // Remove manual description if empty.
            $this->descriptionStorage->delete($nid);
          }
        }
      }
    }

    $this->messenger()->addStatus($this->t('Descriptions saved successfully.'));
  }

  /**
   * AJAX callback to reformulate a description with AI.
   */
  public function reformulateDescriptionAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    try {
      // Get the triggering element to retrieve data attributes and form path.
      $triggering_element = $form_state->getTriggeringElement();
      $parents = $triggering_element['#array_parents'];

      // Get data from button attributes.
      $nid = $triggering_element['#attributes']['data-nid'];
      $title = $triggering_element['#attributes']['data-title'];

      // Determine if this is homepage or menu table.
      // Homepage parents: ['homepage_table', 0, 'manual', 'ai_button']
      // Menu parents: ['menu_MENUID', ROW_INDEX, 'manual', 'ai_button']
      $is_homepage = ($parents[0] === 'homepage_table');

      if ($is_homepage) {
        // Get current textarea content from homepage table.
        $current_content = $form_state->getValue(['homepage_table', 0, 'manual', 'textarea']) ?? '';
      }
      else {
        // Get current textarea content from menu table.
        $menu_key = $parents[0];
        $row_index = $parents[1];
        $current_content = $form_state->getValue([$menu_key, $row_index, 'manual', 'textarea']) ?? '';
      }

      // Build context for AI reformulation: title + current content.
      $context_parts = [];
      $context_parts[] = "Page title: " . $title;
      if (!empty($current_content)) {
        $context_parts[] = "Current description: " . $current_content;
      }

      $context = implode("\n", $context_parts);

      // Reformulate with AI using title + current content as context.
      $reformulated = $this->aiReformulation->reformulate($context, 'description');

      // Update the textarea value using data-nid selector (KISS: simple and reliable).
      // This works for both homepage and menu tables since they both use data-nid attribute.
      $response->addCommand(new InvokeCommand('textarea[data-nid="' . $nid . '"]', 'val', [$reformulated]));

      $this->messenger()->addStatus($this->t('AI suggestion generated. Please review and modify as needed before saving.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to generate AI suggestion: @message', ['@message' => $e->getMessage()]));
    }

    return $response;
  }

  /**
   * Gets a status badge for display.
   *
   * @param string $status
   *   The status string.
   *
   * @return string
   *   HTML badge markup.
   */
  protected function getStatusBadge(string $status): string {
    $badges = [
      'new' => '<span style="background: #2196F3; color: white; padding: 2px 8px; border-radius: 3px; font-size: 12px;">🆕 New</span>',
      'modified' => '<span style="background: #FF9800; color: white; padding: 2px 8px; border-radius: 3px; font-size: 12px;">⚠️ Modified</span>',
      'unchanged' => '<span style="background: #4CAF50; color: white; padding: 2px 8px; border-radius: 3px; font-size: 12px;">✅ Up to date</span>',
      'manual_override' => '<span style="background: #9E9E9E; color: white; padding: 2px 8px; border-radius: 3px; font-size: 12px;">📝 Manual</span>',
    ];

    return $badges[$status] ?? $status;
  }

  /**
   * Gets the final description that will be used in llms.txt.
   *
   * This follows the same priority logic as LlmsTxtGeneratorService::getDescription():
   * 1. Manual override
   * 2. Reformulated description
   * 3. Fallback to title
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string $fallback_title
   *   Fallback title if no description found.
   *
   * @return array
   *   Array with keys:
   *   - 'text': The description text
   *   - 'source': The source ('manual', 'reformulated', or 'title')
   */
  protected function getFinalDescription(int|string $nid, string $fallback_title): array {
    // Priority 1: Manual override.
    $manual = $this->descriptionStorage->get($nid);
    if (!empty($manual)) {
      return [
        'text' => $manual,
        'source' => 'manual',
      ];
    }

    // Priority 2: Reformulated description.
    $reformulated_data = $this->reformulatedStorage->get($nid);
    if (!empty($reformulated_data['reformulated'])) {
      return [
        'text' => $reformulated_data['reformulated'],
        'source' => 'reformulated',
      ];
    }

    // Priority 3: Fallback to title.
    return [
      'text' => $fallback_title,
      'source' => 'title',
    ];
  }

  /**
   * Formats the final description for display.
   *
   * @param array $final_description
   *   Array with 'text' and 'source' keys.
   *
   * @return string
   *   Formatted HTML markup.
   */
  protected function formatFinalDescription(array $final_description): string {
    $text = $final_description['text'];
    $source = $final_description['source'];

    $source_labels = [
      'manual' => '<small style="color: #9E9E9E;">📝 ' . $this->t('Manual') . '</small>',
      'reformulated' => '<small style="color: #4CAF50;">🤖 ' . $this->t('AI Reformulated') . '</small>',
      'title' => '<small style="color: #FF9800;">⚠️ ' . $this->t('Fallback (Title)') . '</small>',
    ];

    $source_label = $source_labels[$source] ?? '';

    return '<div style="background: #f5f5f5; padding: 8px; border-radius: 3px; border-left: 3px solid #ddd;">' .
           '<div>' . htmlspecialchars($text) . '</div>' .
           '<div style="margin-top: 4px;">' . $source_label . '</div>' .
           '</div>';
  }

}

