<?php

namespace Drupal\llms_txt_ai\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\llms_txt_ai\Service\PromptManagerService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for AI prompts.
 */
class PromptsForm extends ConfigFormBase {

  /**
   * The prompt manager service.
   *
   * @var \Drupal\llms_txt_ai\Service\PromptManagerService
   */
  protected $promptManager;

  /**
   * Constructs a PromptsForm object.
   *
   * @param \Drupal\llms_txt_ai\Service\PromptManagerService $prompt_manager
   *   The prompt manager service.
   */
  public function __construct(PromptManagerService $prompt_manager) {
    $this->promptManager = $prompt_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('llms_txt_ai.prompt_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['llms_txt_ai.prompts'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'llms_txt_ai_prompts_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['#attached']['library'][] = 'llms_txt_ai/prompts_admin';

    // Get system and user prompts.
    $system_prompts = $this->promptManager->getSystemPrompts();
    $user_prompts = $this->promptManager->getUserPrompts();

    // Build status list.
    $status_items = [];
    $contexts = [
      'summary' => $this->t('Summary'),
      'introduction' => $this->t('Introduction'),
      'description' => $this->t('Menu Description'),
    ];

    foreach ($contexts as $context => $label) {
      $is_custom = !empty($user_prompts[$context]);
      $status_text = $is_custom ? $this->t('Custom Prompt') : $this->t('System Prompt');
      $status_class = $is_custom ? 'status-custom' : 'status-system';
      $status_items[] = '<li><strong>' . $label . ':</strong> <span class="' . $status_class . '">' . $status_text . '</span></li>';
    }

    // Status at the top.
    $form['status'] = [
      '#type' => 'markup',
      '#markup' => '<div class="prompts-status"><strong>' . $this->t('Status:') . '</strong><ul>' .
        implode('', $status_items) .
        '</ul></div>',
    ];

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure AI prompts for text reformulation. Leave custom prompts empty to use system defaults.') . '</p>',
    ];

    // Summary section.
    $form['summary_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Summary Prompt'),
      '#open' => TRUE,
    ];

    $form['summary_section']['system_summary'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Prompt (read-only)'),
      '#default_value' => $system_prompts['summary']['content'] ?? '',
      '#rows' => 8,
      '#disabled' => TRUE,
      '#attributes' => ['readonly' => 'readonly', 'class' => ['system-prompt']],
    ];

    $form['summary_section']['copy_summary'] = [
      '#type' => 'button',
      '#value' => $this->t('📋 Copy to Custom'),
      '#attributes' => [
        'onclick' => 'document.getElementById("edit-custom-summary").value = document.getElementById("edit-system-summary").value; return false;',
        'class' => ['copy-button'],
      ],
    ];

    $form['summary_section']['custom_summary'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Prompt (leave empty to use system)'),
      '#default_value' => $user_prompts['summary'] ?? '',
      '#rows' => 8,
      '#attributes' => ['class' => ['custom-prompt']],
    ];

    // Introduction section.
    $form['introduction_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Introduction Prompt'),
      '#open' => TRUE,
    ];

    $form['introduction_section']['system_introduction'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Prompt (read-only)'),
      '#default_value' => $system_prompts['introduction']['content'] ?? '',
      '#rows' => 8,
      '#disabled' => TRUE,
      '#attributes' => ['readonly' => 'readonly', 'class' => ['system-prompt']],
    ];

    $form['introduction_section']['copy_introduction'] = [
      '#type' => 'button',
      '#value' => $this->t('📋 Copy to Custom'),
      '#attributes' => [
        'onclick' => 'document.getElementById("edit-custom-introduction").value = document.getElementById("edit-system-introduction").value; return false;',
        'class' => ['copy-button'],
      ],
    ];

    $form['introduction_section']['custom_introduction'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Prompt (leave empty to use system)'),
      '#default_value' => $user_prompts['introduction'] ?? '',
      '#rows' => 8,
      '#attributes' => ['class' => ['custom-prompt']],
    ];

    // Description section.
    $form['description_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Menu Description Prompt'),
      '#open' => TRUE,
    ];

    $form['description_section']['system_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Prompt (read-only)'),
      '#default_value' => $system_prompts['description']['content'] ?? '',
      '#rows' => 8,
      '#disabled' => TRUE,
      '#attributes' => ['readonly' => 'readonly', 'class' => ['system-prompt']],
    ];

    $form['description_section']['copy_description'] = [
      '#type' => 'button',
      '#value' => $this->t('📋 Copy to Custom'),
      '#attributes' => [
        'onclick' => 'document.getElementById("edit-custom-description").value = document.getElementById("edit-system-description").value; return false;',
        'class' => ['copy-button'],
      ],
    ];

    $form['description_section']['custom_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom Prompt (leave empty to use system)'),
      '#default_value' => $user_prompts['description'] ?? '',
      '#rows' => 8,
      '#attributes' => ['class' => ['custom-prompt']],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Save custom prompts (empty values will use system prompts).
    $this->promptManager->saveUserPrompt('summary', $form_state->getValue('custom_summary'));
    $this->promptManager->saveUserPrompt('introduction', $form_state->getValue('custom_introduction'));
    $this->promptManager->saveUserPrompt('description', $form_state->getValue('custom_description'));

    parent::submitForm($form, $form_state);
  }

}

