<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service to reformulate text using AI.
 */
class AiReformulationService {

  /**
   * Default prompt for summary reformulation.
   */
  const DEFAULT_PROMPT_SUMMARY = <<<'EOT'
<context>
You are helping create content for an llms.txt file - a standardized format that helps AI assistants understand websites.
The summary is the first thing AI assistants read to quickly understand what this website is about.
</context>

<instructions>
Write a concise, informative summary that:
- Is exactly 1-2 sentences (maximum 50 words)
- Clearly states what the website does or offers
- Uses simple, direct language
- Focuses on the core value proposition
- Avoids marketing jargon and buzzwords
- Is written in third person
</instructions>

<constraints>
- Maximum length: 2 sentences or 50 words
- No promotional language ("best", "leading", "revolutionary")
- No vague terms ("innovative", "cutting-edge", "world-class")
- No calls to action
- Focus on facts, not opinions
</constraints>

<input_text>
{text}
</input_text>

<output_format>
Return ONLY the rewritten summary text, without any preamble, explanation, or meta-commentary.
</output_format>
EOT;

  /**
   * Default prompt for introduction reformulation.
   */
  const DEFAULT_PROMPT_INTRODUCTION = <<<'EOT'
<context>
You are helping create content for an llms.txt file - a standardized format that helps AI assistants understand websites.
The introduction provides a comprehensive overview that helps AI assistants deeply understand the website's purpose, audience, and value.
</context>

<instructions>
Rewrite this introduction to be clear, informative, and well-structured:
- Write 2-3 concise paragraphs (150-250 words total)
- First paragraph: What the website is and its primary purpose
- Second paragraph: Who it's for and what problems it solves
- Third paragraph (optional): Key features, approach, or unique aspects
- Use natural, conversational language
- Be specific and concrete, not vague or abstract
- Focus on information that helps AI assistants understand context
</instructions>

<constraints>
- Length: 2-3 paragraphs, 150-250 words total
- No marketing hyperbole or promotional language
- No bullet points or lists (use flowing paragraphs)
- No calls to action or sales pitches
- Avoid jargon unless necessary (then explain it)
- Write in third person
</constraints>

<tone>
Professional yet approachable, informative without being dry, clear without being simplistic.
</tone>

<input_text>
{text}
</input_text>

<output_format>
Return ONLY the rewritten introduction text in paragraph form, without any preamble, explanation, or meta-commentary.
</output_format>
EOT;

  /**
   * Default prompt for description reformulation.
   */
  const DEFAULT_PROMPT_DESCRIPTION = <<<'EOT'
<context>
You are helping create content for an llms.txt file - a standardized format that helps AI assistants understand websites.
This description is derived from SEO meta descriptions and should provide a clear, concise explanation of a specific page or section.
</context>

<instructions>
Transform this SEO meta description into natural, AI-friendly language:
- Keep it concise: 1-2 sentences maximum
- Remove SEO-optimized keywords stuffing
- Make it conversational and easy to understand
- Focus on what the page actually contains or does
- Preserve the core information while improving clarity
- Use active voice when possible
</instructions>

<constraints>
- Maximum length: 2 sentences or 40 words
- No keyword stuffing or SEO tricks
- No promotional language or superlatives
- No calls to action ("Learn more", "Contact us", etc.)
- No special characters or emojis
</constraints>

<examples>
BAD: "Discover our amazing products! Best prices guaranteed. Shop now for exclusive deals on top-quality items. Free shipping available!"
GOOD: "An online store offering electronics, home goods, and accessories with free shipping on orders over $50."

BAD: "Leading innovative solutions for enterprise-level digital transformation and cutting-edge technology implementation."
GOOD: "Provides software consulting services to help businesses modernize their technology infrastructure."
</examples>

<input_text>
{text}
</input_text>

<output_format>
Return ONLY the rewritten description text, without any preamble, explanation, or meta-commentary.
</output_format>
EOT;

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The prompt manager service.
   *
   * @var \Drupal\llms_txt_ai\Service\PromptManagerService
   */
  protected $promptManager;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an AiReformulationService object.
   *
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider manager.
   * @param \Drupal\llms_txt_ai\Service\PromptManagerService $prompt_manager
   *   The prompt manager service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    AiProviderPluginManager $ai_provider_manager,
    PromptManagerService $prompt_manager,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->aiProviderManager = $ai_provider_manager;
    $this->promptManager = $prompt_manager;
    $this->logger = $logger_factory->get('llms_txt_ai');
  }

  /**
   * Reformulates text using AI.
   *
   * @param string $text
   *   The text to reformulate.
   * @param string $context
   *   Context type: 'description' or 'introduction'.
   *
   * @return string
   *   The reformulated text, or original text if AI fails.
   */
  public function reformulate(string $text, string $context = 'description'): string {
    try {
      // Get the default provider and model for chat operation.
      $operation_type = 'chat';
      $default = $this->aiProviderManager->getDefaultProviderForOperationType($operation_type);

      if (empty($default['provider_id']) || empty($default['model_id'])) {
        $this->logger->error('No default provider or model set for chat operation. Please configure AI settings at /admin/config/ai.');
        return $text;
      }

      // Create provider instance.
      $provider = $this->aiProviderManager->createInstance($default['provider_id']);
      $model_id = $default['model_id'];

      // Build prompt based on context.
      $prompt = $this->buildPrompt($text, $context);

      // Create ChatInput with ChatMessage.
      $message = new ChatMessage('user', $prompt);
      $input = new ChatInput([$message]);

      // Call AI provider with proper ChatInput object and model ID.
      $output = $provider->chat($input, $model_id);

      // Extract normalized text from ChatOutput.
      if (!empty($output)) {
        $normalized = $output->getNormalized();
        if ($normalized) {
          return trim($normalized->getText());
        }
      }

      return $text;
    }
    catch (\Exception $e) {
      $this->logger->error('AI reformulation failed: @message', ['@message' => $e->getMessage()]);
      return $text;
    }
  }

  /**
   * Builds the AI prompt based on context.
   *
   * @param string $text
   *   The text to reformulate.
   * @param string $context
   *   Context type: 'description', 'introduction', or 'summary'.
   *
   * @return string
   *   The prompt for the AI.
   */
  protected function buildPrompt(string $text, string $context): string {
    // Get the prompt content from the prompt manager.
    // This will use user prompt if available, otherwise system prompt.
    $prompt_template = $this->promptManager->getPromptContent($context);

    // Fallback to default constants if no prompt found.
    if (empty($prompt_template)) {
      if ($context === 'summary') {
        $prompt_template = self::DEFAULT_PROMPT_SUMMARY;
      }
      elseif ($context === 'introduction') {
        $prompt_template = self::DEFAULT_PROMPT_INTRODUCTION;
      }
      else {
        $prompt_template = self::DEFAULT_PROMPT_DESCRIPTION;
      }
    }

    // Replace {text} placeholder with actual text.
    return str_replace('{text}', $text, $prompt_template);
  }

}

