<?php

namespace Drupal\llms_txt_ai\Service;

/**
 * Service to detect changes in page descriptions.
 */
class ChangeDetectionService {

  /**
   * The description storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\DescriptionStorageService
   */
  protected $descriptionStorage;

  /**
   * The reformulated storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\ReformulatedStorageService
   */
  protected $reformulatedStorage;

  /**
   * The meta description service.
   *
   * @var \Drupal\llms_txt_ai\Service\MetaDescriptionService
   */
  protected $metaDescriptionService;

  /**
   * Constructs a ChangeDetectionService object.
   *
   * @param \Drupal\llms_txt_ai\Service\DescriptionStorageService $description_storage
   *   The description storage service.
   * @param \Drupal\llms_txt_ai\Service\ReformulatedStorageService $reformulated_storage
   *   The reformulated storage service.
   * @param \Drupal\llms_txt_ai\Service\MetaDescriptionService $meta_description_service
   *   The meta description service.
   */
  public function __construct(
    DescriptionStorageService $description_storage,
    ReformulatedStorageService $reformulated_storage,
    MetaDescriptionService $meta_description_service
  ) {
    $this->descriptionStorage = $description_storage;
    $this->reformulatedStorage = $reformulated_storage;
    $this->metaDescriptionService = $meta_description_service;
  }

  /**
   * Detects changes in pages and classifies them.
   *
   * @param array $pages
   *   Array of pages from MenuExtractorService.
   *
   * @return array
   *   Array with keys: to_reformulate, unchanged, manual_override, to_remove.
   */
  public function detectChanges(array $pages): array {
    $result = [
      'to_reformulate' => [],
      'unchanged' => [],
      'manual_override' => [],
      'to_remove' => [],
    ];

    $current_nids = [];

    foreach ($pages as $page) {
      $nid = $page['nid'];
      $current_nids[] = $nid;

      // Check if manual override exists.
      $manual_description = $this->descriptionStorage->get($nid);
      if (!empty($manual_description)) {
        $result['manual_override'][] = $nid;
        continue;
      }

      // Get source description (meta or title).
      $source_text = $this->getSourceDescription($page);
      $source_hash = md5($source_text);

      // Check if reformulated version exists.
      $reformulated_data = $this->reformulatedStorage->get($nid);

      if (!$reformulated_data) {
        // New page - needs reformulation.
        $result['to_reformulate'][] = $nid;
      }
      elseif ($reformulated_data['source_hash'] !== $source_hash) {
        // Source changed - needs reformulation.
        $result['to_reformulate'][] = $nid;
      }
      else {
        // Unchanged - reuse existing reformulation.
        $result['unchanged'][] = $nid;
      }
    }

    // Find orphaned entries (in storage but not in current menus).
    $all_reformulated = $this->reformulatedStorage->getAll();
    foreach (array_keys($all_reformulated) as $stored_nid) {
      if (!in_array((int) $stored_nid, $current_nids)) {
        $result['to_remove'][] = (int) $stored_nid;
      }
    }

    return $result;
  }

  /**
   * Gets the source description for a page.
   *
   * @param array $page
   *   Page array with keys: title, url, nid.
   *
   * @return string
   *   The source description (meta or title).
   */
  protected function getSourceDescription(array $page): string {
    $meta = $this->metaDescriptionService->getMetaDescription($page['nid']);
    return !empty($meta) ? $meta : $page['title'];
  }

  /**
   * Gets the status for a specific page.
   *
   * @param array $page
   *   Page array with keys: title, url, nid.
   *
   * @return string
   *   Status: 'new', 'modified', 'unchanged', 'manual_override'.
   */
  public function getPageStatus(array $page): string {
    $nid = $page['nid'];

    // Check manual override.
    if (!empty($this->descriptionStorage->get($nid))) {
      return 'manual_override';
    }

    // Get source and check reformulated.
    $source_text = $this->getSourceDescription($page);
    $source_hash = md5($source_text);
    $reformulated_data = $this->reformulatedStorage->get($nid);

    if (!$reformulated_data) {
      return 'new';
    }

    if ($reformulated_data['source_hash'] !== $source_hash) {

      return 'modified';
    }

    return 'unchanged';
  }

}

