<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\State\StateInterface;

/**
 * Service to generate the final llms.txt file.
 */
class LlmsTxtGeneratorService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The description storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\DescriptionStorageService
   */
  protected $descriptionStorage;

  /**
   * The reformulated storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\ReformulatedStorageService
   */
  protected $reformulatedStorage;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * Constructs a LlmsTxtGeneratorService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\llms_txt_ai\Service\MenuExtractorService $menu_extractor
   *   The menu extractor service.
   * @param \Drupal\llms_txt_ai\Service\DescriptionStorageService $description_storage
   *   The description storage service.
   * @param \Drupal\llms_txt_ai\Service\ReformulatedStorageService $reformulated_storage
   *   The reformulated storage service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $url_generator
   *   The URL generator.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    MenuExtractorService $menu_extractor,
    DescriptionStorageService $description_storage,
    ReformulatedStorageService $reformulated_storage,
    CacheBackendInterface $cache,
    UrlGeneratorInterface $url_generator,
    StateInterface $state
  ) {
    $this->configFactory = $config_factory;
    $this->menuExtractor = $menu_extractor;
    $this->descriptionStorage = $description_storage;
    $this->reformulatedStorage = $reformulated_storage;
    $this->cache = $cache;
    $this->urlGenerator = $url_generator;
    $this->state = $state;
  }

  /**
   * Generates the llms.txt content.
   *
   * @return string
   *   The llms.txt content.
   */
  public function generate(): string {
    $config = $this->configFactory->get('llms_txt_ai.settings');

    // Get configuration.
    $summary = $config->get('summary') ?? '';
    $introduction = $config->get('introduction') ?? '';
    $menus = $config->get('menus') ?? [];
    $include_homepage = $config->get('include_homepage') ?? TRUE;

    // Build content.
    $content = [];
    $base_url = \Drupal::request()->getSchemeAndHttpHost();

    // Site name (H1).
    $site_name = $this->configFactory->get('system.site')->get('name');
    $content[] = "# {$site_name}";
    $content[] = '';

    // Summary (blockquote).
    if (!empty($summary)) {
      $content[] = "> {$summary}";
      $content[] = '';
    }

    // Introduction (optional details).
    if (!empty($introduction)) {
      $content[] = $introduction;
      $content[] = '';
    }

    // Collect all NIDs for cleanup.
    $all_nids = [];

    // Homepage (before sections if exists).
    if ($include_homepage) {
      $homepage = $this->menuExtractor->getHomepage();
      if ($homepage) {
        $nid = $homepage['nid'];
        $title = $homepage['title'];
        $url = $base_url . $homepage['url'];
        $description = $this->getDescription($nid, $title);

        $content[] = "- [{$title}]({$url}): {$description}";
        $content[] = '';
        $all_nids[] = $nid;
      }
    }

    // Extract pages grouped by menu.
    $menus_data = $this->menuExtractor->extractPagesGroupedByMenu($menus);

    // Generate sections (H2) for each menu.
    foreach ($menus_data as $menu_data) {
      $menu_name = $menu_data['menu_name'];
      $pages = $menu_data['pages'];

      if (empty($pages)) {
        continue;
      }

      $content[] = "## {$menu_name}";
      $content[] = '';

      foreach ($pages as $page) {
        $nid = $page['nid'];
        $title = $page['title'];
        $url = $base_url . $page['url'];
        $description = $this->getDescription($nid, $title);

        $content[] = "- [{$title}]({$url}): {$description}";
        $all_nids[] = $nid;
      }

      $content[] = '';
    }

    $content[] = '---';
    $content[] = 'Generated by llms.txt AI Generator for Drupal';

    $output = implode("\n", $content);

    // Store in State API (persists across cache clears).
    $this->state->set('llms_txt_ai.generated_content', $output);
    $this->state->set('llms_txt_ai.generated_timestamp', time());

    // Also cache for performance (but State is the source of truth).
    $this->cache->set('llms_txt_ai:output', $output, CacheBackendInterface::CACHE_PERMANENT, ['llms_txt_ai_output']);

    // Cleanup orphaned entries.
    $this->reformulatedStorage->cleanup($all_nids);

    return $output;
  }

  /**
   * Gets the description for a page.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string $fallback_title
   *   Fallback title if no description found.
   *
   * @return string
   *   The description.
   */
  protected function getDescription(int|string $nid, string $fallback_title): string {
    // Priority 1: Manual override.
    $manual = $this->descriptionStorage->get($nid);
    if (!empty($manual)) {
      return $manual;
    }

    // Priority 2: Reformulated description.
    $reformulated_data = $this->reformulatedStorage->get($nid);
    if (!empty($reformulated_data['reformulated'])) {
      return $reformulated_data['reformulated'];
    }

    // Priority 3: Fallback to title.
    return $fallback_title;
  }

  /**
   * Gets llms.txt content.
   *
   * First tries cache for performance, then falls back to State API.
   * State API persists across cache clears.
   *
   * @return string|null
   *   Content or NULL if not generated.
   */
  public function getCached(): ?string {
    // Try cache first for performance.
    $cached = $this->cache->get('llms_txt_ai:output');
    if ($cached) {
      return $cached->data;
    }

    // Fall back to State API (survives cache clears).
    $content = $this->state->get('llms_txt_ai.generated_content');

    // If found in State, restore to cache for next time.
    if (!empty($content)) {
      $this->cache->set('llms_txt_ai:output', $content, CacheBackendInterface::CACHE_PERMANENT, ['llms_txt_ai_output']);
      return $content;
    }

    return NULL;
  }

  /**
   * Deletes the generated content.
   */
  public function invalidateCache(): void {
    $this->cache->delete('llms_txt_ai:output');
    $this->state->delete('llms_txt_ai.generated_content');
    $this->state->delete('llms_txt_ai.generated_timestamp');
  }

  /**
   * Gets the timestamp of last generation.
   *
   * @return int|null
   *   Unix timestamp or NULL if never generated.
   */
  public function getGeneratedTimestamp(): ?int {
    return $this->state->get('llms_txt_ai.generated_timestamp');
  }

}

