<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Menu\MenuLinkTreeInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Drupal\path_alias\AliasManagerInterface;

/**
 * Service to extract pages from Drupal menus.
 */
class MenuExtractorService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTreeInterface
   */
  protected $menuLinkTree;

  /**
   * The path alias manager.
   *
   * @var \Drupal\path_alias\AliasManagerInterface
   */
  protected $pathAliasManager;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a MenuExtractorService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Menu\MenuLinkTreeInterface $menu_link_tree
   *   The menu link tree service.
   * @param \Drupal\path_alias\AliasManagerInterface $path_alias_manager
   *   The path alias manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, MenuLinkTreeInterface $menu_link_tree, AliasManagerInterface $path_alias_manager, ConfigFactoryInterface $config_factory) {
    $this->entityTypeManager = $entity_type_manager;
    $this->menuLinkTree = $menu_link_tree;
    $this->pathAliasManager = $path_alias_manager;
    $this->configFactory = $config_factory;
  }

  /**
   * Gets the configured depth for a specific menu.
   *
   * @param string $menu_id
   *   The menu ID.
   *
   * @return int
   *   The depth to use for this menu.
   */
  public function getMenuDepth(string $menu_id): int {
    $config = $this->configFactory->get('llms_txt_ai.settings');

    // Priority 1: Menu-specific depth.
    $menu_depths = $config->get('menu_depths') ?? [];
    if (isset($menu_depths[$menu_id])) {
      return (int) $menu_depths[$menu_id];
    }

    // Priority 2: Global max depth.
    $max_depth = $config->get('menu_depth') ?? 3;
    return (int) $max_depth;
  }

  /**
   * Sorts menu IDs according to configured order.
   *
   * @param array $menu_ids
   *   Array of menu IDs to sort.
   *
   * @return array
   *   Sorted array of menu IDs.
   */
  protected function sortMenusByOrder(array $menu_ids): array {
    $config = $this->configFactory->get('llms_txt_ai.settings');
    $menu_order = $config->get('menu_order') ?? [];

    // If no order is configured, return menus as-is.
    if (empty($menu_order)) {
      return $menu_ids;
    }

    // Create a sorted array based on menu_order.
    $sorted = [];
    foreach ($menu_order as $menu_id) {
      if (in_array($menu_id, $menu_ids)) {
        $sorted[] = $menu_id;
      }
    }

    // Add any menus that are not in the order configuration.
    foreach ($menu_ids as $menu_id) {
      if (!in_array($menu_id, $sorted)) {
        $sorted[] = $menu_id;
      }
    }

    return $sorted;
  }

  /**
   * Extracts pages from selected menus.
   *
   * @param array $menu_ids
   *   Array of menu IDs to extract from.
   *
   * @return array
   *   Array of pages with keys: title, url, nid.
   */
  public function extractPages(array $menu_ids): array {
    $pages = [];

    // Sort menus by configured order.
    $menu_ids = $this->sortMenusByOrder($menu_ids);

    foreach ($menu_ids as $menu_id) {
      if (empty($menu_id)) {
        continue;
      }

      $depth = $this->getMenuDepth($menu_id);

      $parameters = new MenuTreeParameters();
      $parameters->setMaxDepth($depth);
      $parameters->onlyEnabledLinks();

      $tree = $this->menuLinkTree->load($menu_id, $parameters);
      $manipulators = [
        ['callable' => 'menu.default_tree_manipulators:checkAccess'],
        ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
      ];
      $tree = $this->menuLinkTree->transform($tree, $manipulators);

      $this->extractPagesFromTree($tree, $pages);
    }

    // Remove duplicates based on NID.
    $unique_pages = [];
    $seen_nids = [];
    foreach ($pages as $page) {
      if (!empty($page['nid']) && !in_array($page['nid'], $seen_nids)) {
        $unique_pages[] = $page;
        $seen_nids[] = $page['nid'];
      }
    }

    return $unique_pages;
  }

  /**
   * Extracts pages from selected menus, grouped by menu with hierarchy.
   *
   * @param array $menu_ids
   *   Array of menu IDs to extract from.
   *
   * @return array
   *   Array of menus, each containing:
   *   - menu_id: The menu machine name
   *   - menu_name: The human-readable menu name
   *   - pages: Array of pages with keys: title, url, nid, depth, parent_id
   */
  public function extractPagesGroupedByMenu(array $menu_ids): array {
    $menus = [];
    $menu_storage = $this->entityTypeManager->getStorage('menu');

    // Sort menus by configured order.
    $menu_ids = $this->sortMenusByOrder($menu_ids);

    foreach ($menu_ids as $menu_id) {
      if (empty($menu_id)) {
        continue;
      }

      // Load menu entity to get the label.
      $menu_entity = $menu_storage->load($menu_id);
      if (!$menu_entity) {
        continue;
      }

      $depth = $this->getMenuDepth($menu_id);

      $parameters = new MenuTreeParameters();
      $parameters->setMaxDepth($depth);
      $parameters->onlyEnabledLinks();

      $tree = $this->menuLinkTree->load($menu_id, $parameters);
      $manipulators = [
        ['callable' => 'menu.default_tree_manipulators:checkAccess'],
        ['callable' => 'menu.default_tree_manipulators:generateIndexAndSort'],
      ];
      $tree = $this->menuLinkTree->transform($tree, $manipulators);

      $pages = [];
      $seen_nids = [];
      $this->extractPagesFromTreeWithHierarchy($tree, $pages, $seen_nids, 0);

      // Only add menu if it has pages.
      if (!empty($pages)) {
        $menus[] = [
          'menu_id' => $menu_id,
          'menu_name' => $menu_entity->label(),
          'pages' => $pages,
        ];
      }
    }

    return $menus;
  }

  /**
   * Recursively extracts pages from menu tree.
   *
   * @param array $tree
   *   Menu tree array.
   * @param array &$pages
   *   Reference to pages array to populate.
   */
  protected function extractPagesFromTree(array $tree, array &$pages): void {
    foreach ($tree as $element) {
      $link = $element->link;
      $url = $link->getUrlObject();

      // Only process internal routes.
      if ($url->isRouted() && $url->getRouteName() === 'entity.node.canonical') {
        $route_parameters = $url->getRouteParameters();
        if (!empty($route_parameters['node'])) {
          $nid = $route_parameters['node'];
          $pages[] = [
            'title' => $link->getTitle(),
            'url' => $url->toString(),
            'nid' => $nid,
          ];
        }
      }

      // Process subtree.
      if ($element->hasChildren && !empty($element->subtree)) {
        $this->extractPagesFromTree($element->subtree, $pages);
      }
    }
  }

  /**
   * Recursively extracts pages from menu tree with hierarchy information.
   *
   * @param array $tree
   *   Menu tree array.
   * @param array &$pages
   *   Reference to pages array to populate.
   * @param array &$seen_nids
   *   Reference to array tracking seen NIDs to avoid duplicates.
   * @param int $depth
   *   Current depth level (0-based).
   * @param int|null $parent_id
   *   Parent NID if this is a child item.
   */
  protected function extractPagesFromTreeWithHierarchy(array $tree, array &$pages, array &$seen_nids, int $depth = 0, ?int $parent_id = NULL): void {
    foreach ($tree as $element) {
      $link = $element->link;
      $url = $link->getUrlObject();

      // Only process internal routes.
      if ($url->isRouted() && $url->getRouteName() === 'entity.node.canonical') {
        $route_parameters = $url->getRouteParameters();
        if (!empty($route_parameters['node'])) {
          $nid = $route_parameters['node'];

          // Skip duplicates within this menu.
          if (!in_array($nid, $seen_nids)) {
            $pages[] = [
              'title' => $link->getTitle(),
              'url' => $url->toString(),
              'nid' => $nid,
              'depth' => $depth,
              'parent_id' => $parent_id,
            ];
            $seen_nids[] = $nid;

            // Process subtree with this item as parent.
            if ($element->hasChildren && !empty($element->subtree)) {
              $this->extractPagesFromTreeWithHierarchy($element->subtree, $pages, $seen_nids, $depth + 1, $nid);
            }
          }
        }
      }
      else {
        // For non-node links (like external links or custom routes),
        // still process their children.
        if ($element->hasChildren && !empty($element->subtree)) {
          $this->extractPagesFromTreeWithHierarchy($element->subtree, $pages, $seen_nids, $depth, $parent_id);
        }
      }
    }
  }

  /**
   * Gets the homepage information if it's a node.
   *
   * @return array|null
   *   Homepage data with keys: title, url, nid, is_homepage.
   *   Returns NULL if homepage is not a node.
   */
  public function getHomepage(): ?array {
    // Get the front page path from site configuration.
    $front_page = $this->configFactory->get('system.site')->get('page.front');

    if (empty($front_page)) {
      return NULL;
    }

    // Resolve aliases (e.g., /home -> /node/123).
    $path = $this->pathAliasManager->getPathByAlias($front_page);

    // Extract NID if it's a node.
    if (preg_match('/^\/node\/(\d+)$/', $path, $matches)) {
      $nid = (int) $matches[1];

      // Load the node to get the title.
      $node = $this->entityTypeManager->getStorage('node')->load($nid);

      if ($node && $node->access('view')) {
        return [
          'title' => $node->getTitle(),
          'url' => '/',
          'nid' => $nid,
          'is_homepage' => TRUE,
        ];
      }
    }

    return NULL;
  }

  /**
   * Extracts pages from menus with optional homepage.
   *
   * @param array $menu_ids
   *   Array of menu IDs to extract from.
   * @param bool $include_homepage
   *   Whether to include homepage (default: TRUE).
   *
   * @return array
   *   Array of pages with homepage first if included.
   */
  public function extractPagesWithHomepage(array $menu_ids, bool $include_homepage = TRUE): array {
    // Extract pages from menus (existing logic).
    $pages = $this->extractPages($menu_ids);

    // Add homepage if requested.
    if ($include_homepage) {
      $homepage = $this->getHomepage();
      if ($homepage) {
        $homepage_nid = $homepage['nid'];
        $found = FALSE;

        // Check if homepage NID already exists in pages.
        foreach ($pages as $page) {
          if ($page['nid'] === $homepage_nid) {
            $found = TRUE;
            break;
          }
        }

        // Add homepage at the beginning if not found.
        if (!$found) {
          array_unshift($pages, $homepage);
        }
      }
    }

    return $pages;
  }

}

