<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;

/**
 * Service to retrieve meta descriptions from nodes.
 */
class MetaDescriptionService {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a MetaDescriptionService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler) {
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Gets meta description for a node.
   *
   * Priority order:
   * 1. Metatag field on the entity (field_metatag)
   * 2. Metatag defaults for bundle (node__page)
   * 3. Metatag defaults for entity type (node)
   * 4. Metatag defaults global (global)
   * 5. Body summary field
   *
   * @param int $nid
   *   The node ID.
   *
   * @return string|null
   *   The meta description or NULL if not found.
   */
  public function getMetaDescription(int $nid): ?string {
    try {
      $node = $this->entityTypeManager->getStorage('node')->load($nid);
      if (!$node) {
        return NULL;
      }

      $description_token = NULL;
      $entity_type = $node->getEntityTypeId();
      $bundle = $node->bundle();

      // Try to get meta description from Metatag module.
      if ($this->moduleHandler->moduleExists('metatag')) {

        // Priority 1: Check if entity has metatag field with description.
        if ($node->hasField('field_metatag')) {
          $metatag_value = $node->get('field_metatag')->getValue();
          if (!empty($metatag_value[0]['value'])) {
            // Metatag field stores data as JSON string.
            $metatags = json_decode($metatag_value[0]['value'], TRUE);
            if (!empty($metatags['description'])) {
              $description_token = $metatags['description'];
            }
          }
        }

        // Priority 2-4: If no description from entity field, check defaults.
        if (!$description_token) {
          $metatag_defaults_storage = $this->entityTypeManager->getStorage('metatag_defaults');
          $defaults_ids = [
            $entity_type . '__' . $bundle,  // Priority 2: node__page
            $entity_type,                    // Priority 3: node
            'global',                        // Priority 4: global
          ];

          foreach ($defaults_ids as $defaults_id) {
            $defaults = $metatag_defaults_storage->load($defaults_id);
            if ($defaults && $defaults->status()) {
              $tags = $defaults->get('tags');
              if (!empty($tags['description'])) {
                $description_token = $tags['description'];
                break;
              }
            }
          }
        }

        // If we found a description (from entity or defaults), replace tokens.
        if ($description_token) {
          $token_service = \Drupal::token();
          $description = $token_service->replace($description_token, [$entity_type => $node], ['clear' => TRUE]);

          // Decode HTML entities.
          $description = html_entity_decode($description, ENT_QUOTES | ENT_HTML5);

          // Return if not empty after token replacement.
          $description = trim($description);
          if (!empty($description)) {
            return $description;
          }
        }
      }

      // Priority 5: Fallback to body summary.
      if ($node->hasField('body')) {
        $body = $node->get('body')->getValue();
        if (!empty($body[0]['summary'])) {
          return trim(strip_tags($body[0]['summary']));
        }
      }

      return NULL;
    }
    catch (\Exception $e) {
      // Log error.
      \Drupal::logger('llms_txt_ai')->error('Error getting meta description for node @nid: @message', [
        '@nid' => $nid,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}

