<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Service to manage AI prompts (system and user prompts).
 */
class PromptManagerService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Cache for system prompts.
   *
   * @var array
   */
  protected $systemPrompts = NULL;

  /**
   * Constructs a PromptManagerService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ModuleHandlerInterface $module_handler) {
    $this->configFactory = $config_factory;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Get all system prompts (from YAML files).
   *
   * @return array
   *   Array of system prompts, keyed by prompt ID.
   *   Each prompt has: id, name, description, content.
   */
  public function getSystemPrompts(): array {
    if ($this->systemPrompts !== NULL) {
      return $this->systemPrompts;
    }

    $this->systemPrompts = [];
    $module_path = $this->moduleHandler->getModule('llms_txt_ai')->getPath();
    $prompts_dir = DRUPAL_ROOT . '/' . $module_path . '/config/prompts';

    if (!is_dir($prompts_dir)) {
      return $this->systemPrompts;
    }

    $files = glob($prompts_dir . '/*.yml');
    foreach ($files as $file) {
      try {
        $data = Yaml::parseFile($file);
        if (isset($data['id']) && isset($data['content'])) {
          $this->systemPrompts[$data['id']] = [
            'id' => $data['id'],
            'name' => $data['name'] ?? $data['id'],
            'description' => $data['description'] ?? '',
            'content' => $data['content'],
          ];
        }
      }
      catch (\Exception $e) {
        // Skip invalid files.
        continue;
      }
    }

    return $this->systemPrompts;
  }

  /**
   * Get a specific system prompt by ID.
   *
   * @param string $id
   *   The prompt ID (e.g., 'summary', 'introduction', 'description').
   *
   * @return array|null
   *   The prompt data or NULL if not found.
   */
  public function getSystemPrompt(string $id): ?array {
    $prompts = $this->getSystemPrompts();
    return $prompts[$id] ?? NULL;
  }

  /**
   * Get all user prompts (from config).
   *
   * @return array
   *   Array of user prompts, keyed by prompt ID (simple string values).
   */
  public function getUserPrompts(): array {
    $config = $this->configFactory->get('llms_txt_ai.prompts');
    return [
      'summary' => $config->get('custom_summary') ?? '',
      'introduction' => $config->get('custom_introduction') ?? '',
      'description' => $config->get('custom_description') ?? '',
    ];
  }

  /**
   * Save a user prompt.
   *
   * @param string $id
   *   The prompt ID (summary, introduction, description).
   * @param string $content
   *   The prompt content (empty string to use system prompt).
   */
  public function saveUserPrompt(string $id, string $content): void {
    $config = $this->configFactory->getEditable('llms_txt_ai.prompts');
    $config->set('custom_' . $id, $content)->save();
  }

  /**
   * Get the prompt content to use for a given context.
   *
   * Priority: user prompt (if not empty) > system prompt.
   *
   * @param string $context
   *   The context (e.g., 'summary', 'introduction', 'description').
   *
   * @return string
   *   The prompt content.
   */
  public function getPromptContent(string $context): string {
    // First, check if there's a custom user prompt for this context.
    $user_prompts = $this->getUserPrompts();
    if (!empty($user_prompts[$context])) {
      return $user_prompts[$context];
    }

    // Otherwise, use the system prompt.
    $system_prompt = $this->getSystemPrompt($context);
    if ($system_prompt && !empty($system_prompt['content'])) {
      return $system_prompt['content'];
    }

    // Fallback to empty string.
    return '';
  }

}

