<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;

/**
 * Service to store AI-reformulated descriptions.
 */
class ReformulatedStorageService {

  /**
   * The key-value store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface
   */
  protected $storage;

  /**
   * Constructs a ReformulatedStorageService object.
   *
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $key_value_factory
   *   The key-value factory.
   */
  public function __construct(KeyValueFactoryInterface $key_value_factory) {
    $this->storage = $key_value_factory->get('llms_txt_ai_reformulated');
  }

  /**
   * Sets a reformulated description for a node.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param array $data
   *   Array with keys: source_text, source_hash, reformulated, timestamp, ai_provider.
   */
  public function set(int|string $nid, array $data): void {
    $this->storage->set((string) $nid, $data);
  }

  /**
   * Gets a reformulated description for a node.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   *
   * @return array|null
   *   The reformulated data or NULL if not set.
   */
  public function get(int|string $nid): ?array {
    return $this->storage->get((string) $nid);
  }

  /**
   * Gets all reformulated descriptions.
   *
   * @return array
   *   Array of reformulated data keyed by NID.
   */
  public function getAll(): array {
    return $this->storage->getAll();
  }

  /**
   * Deletes a reformulated description.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   */
  public function delete(int|string $nid): void {
    $this->storage->delete((string) $nid);
  }

  /**
   * Cleans up orphaned entries (pages no longer in menus).
   *
   * @param array $current_nids
   *   Array of current NIDs that should be kept.
   */
  public function cleanup(array $current_nids): void {
    $all_data = $this->getAll();
    foreach (array_keys($all_data) as $nid) {
      if (!in_array((int) $nid, $current_nids)) {
        $this->delete((int) $nid);
      }
    }
  }

}

