<?php

namespace Drupal\llms_txt_ai\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to generate llms.txt file.
 */
class GenerateForm extends ConfirmFormBase {

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The meta description service.
   *
   * @var \Drupal\llms_txt_ai\Service\MetaDescriptionService
   */
  protected $metaDescription;

  /**
   * The description storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\DescriptionStorageService
   */
  protected $descriptionStorage;

  /**
   * The change detection service.
   *
   * @var \Drupal\llms_txt_ai\Service\ChangeDetectionService
   */
  protected $changeDetection;

  /**
   * The batch service.
   *
   * @var \Drupal\llms_txt_ai\Service\BatchService
   */
  protected $batchService;

  /**
   * The AI provider plugin manager.
   *
   * @var \Drupal\ai\PluginManager\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->menuExtractor = $container->get('llms_txt_ai.menu_extractor');
    $instance->metaDescription = $container->get('llms_txt_ai.meta_description');
    $instance->descriptionStorage = $container->get('llms_txt_ai.description_storage');
    $instance->changeDetection = $container->get('llms_txt_ai.change_detection');
    $instance->batchService = $container->get('llms_txt_ai.batch');
    $instance->aiProviderManager = $container->get('ai.provider');
    $instance->loggerFactory = $container->get('logger.factory');
    $instance->languageManager = $container->get('language_manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'llms_txt_ai_generate_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Generate llms.txt now?');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('llms_txt_ai.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('llms_txt_ai.settings');
    $menus = $config->get('menus') ?? [];

    // Validation: Check if AI module has a default chat provider configured.
    try {
      $default = $this->aiProviderManager->getDefaultProviderForOperationType('chat');

      if (empty($default['provider_id']) || empty($default['model_id'])) {
        $this->messenger()->addError($this->t('No default AI provider configured for chat operation. Please <a href="@url">configure AI settings</a> first.', [
          '@url' => '/admin/config/ai',
        ]));
        return $form;
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('AI module not properly configured. Please <a href="@url">configure AI settings</a> first.', [
        '@url' => '/admin/config/ai',
      ]));
      return $form;
    }

    if (empty($menus)) {
      $this->messenger()->addWarning($this->t('No menus configured. The file will be generated with introduction only.'));
    }

    // Language selection.
    $languages = $this->languageManager->getLanguages();
    $language_options = [];
    foreach ($languages as $langcode => $language) {
      $language_options[$langcode] = $language->getName();
    }

    $form['language_selection'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('🌍 Language Selection'),
    ];

    $form['language_selection']['generate_all_languages'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Generate for all enabled languages'),
      '#default_value' => count($language_options) > 1,
      '#description' => $this->t('If checked, llms.txt will be generated for all @count enabled languages.', ['@count' => count($language_options)]),
    ];

    $form['language_selection']['langcode'] = [
      '#type' => 'select',
      '#title' => $this->t('Language'),
      '#options' => $language_options,
      '#default_value' => $this->languageManager->getDefaultLanguage()->getId(),
      '#description' => $this->t('Select the language for which to generate llms.txt.'),
      '#states' => [
        'visible' => [
          ':input[name="generate_all_languages"]' => ['checked' => FALSE],
        ],
      ],
    ];

    // Get selected language for preview.
    $selected_langcode = $form_state->getValue('langcode') ?? $this->languageManager->getDefaultLanguage()->getId();

    // Extract pages and detect changes.
    $pages = $this->menuExtractor->extractPages($menus, $selected_langcode);
    $changes = $this->changeDetection->detectChanges($pages, $selected_langcode);

    // Build summary.
    $summary = [];
    $summary[] = '<h3>' . $this->t('📊 Generation Summary') . '</h3>';
    $summary[] = '<ul>';

    $to_reformulate_count = count($changes['to_reformulate']);
    $unchanged_count = count($changes['unchanged']);
    $manual_count = count($changes['manual_override']);
    $to_remove_count = count($changes['to_remove']);

    if ($to_reformulate_count > 0) {
      $summary[] = '<li>🆕 ' . $this->formatPlural($to_reformulate_count, '1 page to reformulate with AI', '@count pages to reformulate with AI') . '</li>';
    }
    if ($unchanged_count > 0) {
      $summary[] = '<li>✅ ' . $this->formatPlural($unchanged_count, '1 page unchanged (reformulation reused)', '@count pages unchanged (reformulation reused)') . '</li>';
    }
    if ($manual_count > 0) {
      $summary[] = '<li>📝 ' . $this->formatPlural($manual_count, '1 page with manual override (no AI)', '@count pages with manual override (no AI)') . '</li>';
    }
    if ($to_remove_count > 0) {
      $summary[] = '<li>🗑️ ' . $this->formatPlural($to_remove_count, '1 page removed from menu', '@count pages removed from menu') . '</li>';
    }

    $summary[] = '</ul>';

    $cost_label = $this->t('Estimated cost:');
    $api_calls = $this->formatPlural($to_reformulate_count, '1 AI API call', '@count AI API calls');
    $summary[] = '<p><strong>💰 ' . $cost_label . '</strong> ' . $api_calls . '</p>';

    $estimated_time = max(5, $to_reformulate_count * 2);
    $time_label = $this->t('Estimated time:');
    $seconds_label = $this->t('seconds');
    $summary[] = '<p><strong>⏱️ ' . $time_label . '</strong> ~' . $estimated_time . ' ' . $seconds_label . '</p>';

    // Check introduction.
    $introduction = $config->get('introduction');
    if (empty($introduction)) {
      $summary[] = '<p><strong>⚠️ ' . $this->t('Warning:') . '</strong> ' . $this->t('No introduction configured. The file will be generated without a descriptive header.') . '</p>';
    }
    else {
      $intro_length = strlen($introduction);
      $summary[] = '<p><strong>✅ ' . $this->t('Introduction:') . '</strong> ' . $this->t('Configured (@count characters)', ['@count' => $intro_length]) . '</p>';
    }

    $form['summary'] = [
      '#markup' => implode("\n", $summary),
    ];

    $form['force_regenerate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Force complete regeneration (reformulate all pages even if unchanged)'),
      '#default_value' => FALSE,
    ];

    // Store data for submission.
    $form_state->setTemporaryValue('pages', $pages);
    $form_state->setTemporaryValue('changes', $changes);

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $force_regenerate = $form_state->getValue('force_regenerate');
    $generate_all_languages = $form_state->getValue('generate_all_languages');

    // Determine which languages to process.
    $languages_to_process = [];
    if ($generate_all_languages) {
      $languages = $this->languageManager->getLanguages();
      $languages_to_process = array_keys($languages);
    }
    else {
      $languages_to_process[] = $form_state->getValue('langcode');
    }

    // Process each language.
    foreach ($languages_to_process as $langcode) {
      $config = $this->config('llms_txt_ai.settings');
      $menus = $config->get('menus') ?? [];

      // Extract pages for this language.
      $pages = $this->menuExtractor->extractPages($menus, $langcode);

      // IMPORTANT: Recalculate changes at submission time to catch any
      // modifications made after the form was built (e.g., if user edited
      // a page in another tab).
      $changes = $this->changeDetection->detectChanges($pages, $langcode);

      // Prepare pages data for batch.
      $pages_to_process = [];

      if ($force_regenerate) {
        // Force regeneration: process all pages.
        foreach ($pages as $page) {
          $nid = $page['nid'];
          // Skip manual overrides.
          if (!$this->descriptionStorage->get($nid, $langcode)) {
            $source_text = $this->getSourceDescription($page, $langcode);
            $pages_to_process[$nid] = [
              'source_text' => $source_text,
              'langcode' => $langcode,
            ];
          }
        }
      }
      else {
        // Normal: only process changed pages.
        foreach ($changes['to_reformulate'] as $nid) {
          $page = $this->findPageByNid($pages, $nid);
          if ($page) {
            $source_text = $this->getSourceDescription($page, $langcode);
            $pages_to_process[$nid] = [
              'source_text' => $source_text,
              'langcode' => $langcode,
            ];
          }
          else {
            // Log when a page is not found.
            $this->loggerFactory->get('llms_txt_ai')->warning('Page NID @nid is in to_reformulate but not found in pages array. Total pages: @count. Language: @lang', [
              '@nid' => $nid,
              '@count' => count($pages),
              '@lang' => $langcode,
            ]);
          }
        }
      }

      // Create and set batch for this language.
      if (!empty($pages_to_process)) {
        $batch = $this->batchService->createBatch($pages_to_process, $langcode);
        batch_set($batch);
      }
    }
  }

  /**
   * Gets source description for a page.
   *
   * @param array $page
   *   Page array.
   * @param string|null $langcode
   *   The language code. If NULL, uses current language.
   *
   * @return string
   *   Source description.
   */
  protected function getSourceDescription(array $page, ?string $langcode = NULL): string {
    $meta = $this->metaDescription->getMetaDescription($page['nid'], $langcode);
    return !empty($meta) ? $meta : $page['title'];
  }

  /**
   * Finds a page by NID.
   *
   * @param array $pages
   *   Array of pages.
   * @param int $nid
   *   Node ID.
   *
   * @return array|null
   *   Page array or NULL.
   */
  protected function findPageByNid(array $pages, int $nid): ?array {
    foreach ($pages as $page) {
      // Use loose comparison to handle string/int type differences.
      if ($page['nid'] == $nid) {
        return $page;
      }
    }
    return NULL;
  }

}
