<?php

namespace Drupal\llms_txt_ai\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Menu\MenuTreeParameters;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form to configure menu depth per menu.
 */
class MenuDepthForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The menu link tree service.
   *
   * @var \Drupal\Core\Menu\MenuLinkTreeInterface
   */
  protected $menuLinkTree;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->menuLinkTree = $container->get('menu.link_tree');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['llms_txt_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'llms_txt_ai_menu_depth_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('llms_txt_ai.settings');
    $selected_menus = $config->get('menus') ?? [];
    $max_depth = $config->get('menu_depth') ?? 3;
    $menu_depths = $config->get('menu_depths') ?? [];
    $menu_order = $config->get('menu_order') ?? [];

    // Check if menus are configured.
    if (empty($selected_menus)) {
      $form['empty'] = [
        '#markup' => '<div class="messages messages--warning">' .
        '<strong>⚠️ ' . $this->t('No menus configured') . '</strong><br>' .
        $this->t('Please select menus in the <a href="@url">Settings tab</a> before configuring their depth and order.', [
          '@url' => '/admin/config/content/llms-txt-ai',
        ]) .
        '</div>',
      ];
      return $form;
    }

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Configure the order and maximum depth for each menu. Drag rows to reorder.') . '</p>',
    ];

    // Info about max depth.
    $form['max_depth_info'] = [
      '#markup' => '<div class="messages messages--status">' .
      '<strong>ℹ️ ' . $this->t('Maximum depth limit: @max levels', ['@max' => $max_depth]) . '</strong><br>' .
      $this->t('This is the global limit set in Settings. Individual menus cannot exceed this depth.') .
      '</div>',
    ];

    // Sort menus by configured order.
    $sorted_menus = $this->sortMenusByOrder($selected_menus, $menu_order);

    // Table for menu configuration with tabledrag.
    $form['menu_depths_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Menu'),
        $this->t('Structure'),
        $this->t('Depth'),
        $this->t('Weight'),
      ],
      '#empty' => $this->t('No menus configured.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'menu-order-weight',
        ],
      ],
    ];

    $menu_storage = $this->entityTypeManager->getStorage('menu');
    $weight = 0;

    foreach ($sorted_menus as $menu_id) {
      if (empty($menu_id)) {
        continue;
      }

      $menu_entity = $menu_storage->load($menu_id);
      if (!$menu_entity) {
        continue;
      }

      $menu_name = $menu_entity->label();
      $detected_levels = $this->detectMenuLevels($menu_id);
      $current_depth = $menu_depths[$menu_id] ?? $max_depth;

      // Add draggable class for tabledrag.
      $form['menu_depths_table'][$menu_id]['#attributes']['class'][] = 'draggable';
      $form['menu_depths_table'][$menu_id]['#weight'] = $weight;

      // Menu name.
      $form['menu_depths_table'][$menu_id]['name'] = [
        '#markup' => '<strong>' . $menu_name . '</strong><br><small>' . $menu_id . '</small>',
      ];

      // Detected structure.
      $form['menu_depths_table'][$menu_id]['structure'] = [
        '#markup' => $this->formatPlural($detected_levels, '1 level detected', '@count levels detected'),
      ];

      // Depth select.
      $options = [];
      for ($i = 1; $i <= $max_depth; $i++) {
        $options[$i] = $this->formatPlural($i, '1 level', '@count levels');
      }

      $form['menu_depths_table'][$menu_id]['depth'] = [
        '#type' => 'select',
        '#options' => $options,
        '#default_value' => $current_depth,
      ];

      // Weight field for tabledrag.
      $form['menu_depths_table'][$menu_id]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @title', ['@title' => $menu_name]),
        '#title_display' => 'invisible',
        '#default_value' => $weight,
        '#attributes' => ['class' => ['menu-order-weight']],
      ];

      $weight++;
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('llms_txt_ai.settings');
    $selected_menus = $config->get('menus') ?? [];
    $menu_depths = [];
    $menu_order = [];

    // Get values from table.
    $table_values = $form_state->getValue('menu_depths_table');

    // Sort by weight to get the new order.
    uasort($table_values, function ($a, $b) {
      return $a['weight'] <=> $b['weight'];
    });

    foreach ($table_values as $menu_id => $values) {
      if (empty($menu_id) || !in_array($menu_id, $selected_menus)) {
        continue;
      }

      $depth = (int) $values['depth'];
      $menu_depths[$menu_id] = $depth;
      $menu_order[] = $menu_id;
    }

    // Save configuration.
    $config
      ->set('menu_depths', $menu_depths)
      ->set('menu_order', $menu_order)
      ->save();

    $this->messenger()->addStatus($this->t('Menu configuration saved.'));

    parent::submitForm($form, $form_state);
  }

  /**
   * Sorts menus by configured order.
   *
   * @param array $menus
   *   Array of menu IDs.
   * @param array $menu_order
   *   Array of menu IDs in desired order.
   *
   * @return array
   *   Sorted array of menu IDs.
   */
  protected function sortMenusByOrder(array $menus, array $menu_order): array {
    // If no order is configured, return menus as-is.
    if (empty($menu_order)) {
      return $menus;
    }

    // Create a sorted array based on menu_order.
    $sorted = [];
    foreach ($menu_order as $menu_id) {
      if (in_array($menu_id, $menus)) {
        $sorted[] = $menu_id;
      }
    }

    // Add any menus that are not in the order configuration.
    foreach ($menus as $menu_id) {
      if (!in_array($menu_id, $sorted)) {
        $sorted[] = $menu_id;
      }
    }

    return $sorted;
  }

  /**
   * Detects the number of levels in a menu.
   *
   * @param string $menu_id
   *   The menu ID.
   *
   * @return int
   *   The number of levels detected.
   */
  protected function detectMenuLevels(string $menu_id): int {
    $parameters = new MenuTreeParameters();
    $parameters->onlyEnabledLinks();

    $tree = $this->menuLinkTree->load($menu_id, $parameters);

    return $this->getMaxDepth($tree, 1);
  }

  /**
   * Recursively gets the maximum depth of a menu tree.
   *
   * @param array $tree
   *   The menu tree.
   * @param int $current_depth
   *   The current depth level.
   *
   * @return int
   *   The maximum depth.
   */
  protected function getMaxDepth(array $tree, int $current_depth): int {
    $max = $current_depth;

    foreach ($tree as $element) {
      if ($element->hasChildren && !empty($element->subtree)) {
        $child_max = $this->getMaxDepth($element->subtree, $current_depth + 1);
        $max = max($max, $child_max);
      }
    }

    return $max;
  }

}
