<?php

namespace Drupal\llms_txt_ai\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\InvokeCommand;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for llms.txt AI Generator.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The introduction reformulation service.
   *
   * @var \Drupal\llms_txt_ai\Service\IntroductionReformulationService
   */
  protected $introductionReformulation;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->introductionReformulation = $container->get('llms_txt_ai.introduction_reformulation');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['llms_txt_ai.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'llms_txt_ai_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('llms_txt_ai.settings');

    // Introduction section.
    $form['introduction_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Introduction Text'),
      '#open' => TRUE,
    ];

    $form['introduction_section']['summary'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Summary'),
      '#description' => $this->t('A short summary of your site (1-2 sentences). This will appear as a blockquote in llms.txt.'),
      '#default_value' => $config->get('summary'),
      '#rows' => 2,
      '#attributes' => [
        'data-field-type' => 'summary',
      ],
    ];

    $form['introduction_section']['reformulate_summary'] = [
      '#type' => 'button',
      '#value' => $this->t('✨ Reformulate with AI'),
      '#name' => 'reformulate_summary_button',
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::reformulateSummaryAjax',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating summary...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-field-type' => 'summary',
      ],
    ];

    $form['introduction_section']['import_summary'] = [
      '#type' => 'button',
      '#value' => $this->t('📥 Import from homepage'),
      '#name' => 'import_summary_button',
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::importFromHomepageSummaryAjax',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Importing from homepage...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-field-type' => 'summary',
      ],
    ];

    $form['introduction_section']['introduction'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Introduction'),
      '#description' => $this->t('Optional detailed information about your site. This will appear after the summary.'),
      '#default_value' => $config->get('introduction'),
      '#rows' => 6,
      '#attributes' => [
        'data-field-type' => 'introduction',
      ],
    ];

    $form['introduction_section']['reformulate_intro'] = [
      '#type' => 'button',
      '#value' => $this->t('✨ Reformulate with AI'),
      '#name' => 'reformulate_introduction_button',
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::reformulateIntroductionAjax',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Generating introduction...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-field-type' => 'introduction',
      ],
    ];

    $form['introduction_section']['import_intro'] = [
      '#type' => 'button',
      '#value' => $this->t('📥 Import from homepage'),
      '#name' => 'import_introduction_button',
      '#limit_validation_errors' => [],
      '#ajax' => [
        'callback' => '::importFromHomepageIntroductionAjax',
        'event' => 'click',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Importing from homepage...'),
        ],
      ],
      '#attributes' => [
        'class' => ['button', 'button--small'],
        'data-field-type' => 'introduction',
      ],
    ];

    // Menu selection section.
    $form['menu_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Content Sources'),
      '#open' => TRUE,
    ];

    $menu_options = $this->getMenuOptions();
    $form['menu_section']['menus'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Menus to include'),
      '#description' => $this->t('Select which menus to extract pages from.'),
      '#options' => $menu_options,
      '#default_value' => $config->get('menus') ?? [],
    ];

    $form['menu_section']['menu_depth'] = [
      '#type' => 'select',
      '#title' => $this->t('Maximum menu depth'),
      '#description' => $this->t('Maximum depth limit for all menus. You can configure specific depth per menu in the Menu Depth tab.'),
      '#options' => [
        1 => $this->t('1 level'),
        2 => $this->t('2 levels'),
        3 => $this->t('3 levels'),
        4 => $this->t('4 levels'),
        5 => $this->t('5 levels'),
        6 => $this->t('6 levels'),
        7 => $this->t('7 levels'),
        8 => $this->t('8 levels'),
        9 => $this->t('9 levels'),
        10 => $this->t('10 levels'),
      ],
      '#default_value' => $config->get('menu_depth') ?? 3,
    ];

    // Homepage section.
    $form['homepage_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Homepage Settings'),
      '#open' => TRUE,
    ];

    $form['homepage_section']['include_homepage'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include homepage in llms.txt'),
      '#description' => $this->t('Automatically detect and include your site homepage. It will appear first in the pages list.'),
      '#default_value' => $config->get('include_homepage') ?? TRUE,
    ];

    // AI settings section.
    $form['ai_section'] = [
      '#type' => 'details',
      '#title' => $this->t('AI Settings'),
      '#open' => TRUE,
    ];

    $form['ai_section']['info'] = [
      '#markup' => '<p>' . $this->t('This module uses the default AI provider configured in the <a href="@url">AI module settings</a>. Make sure you have configured a default provider for the "Chat" operation type.', [
        '@url' => '/admin/config/ai',
      ]) . '</p>',
    ];

    // Auto-regeneration section.
    $form['auto_regenerate_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Auto-Regeneration'),
      '#open' => TRUE,
    ];

    $form['auto_regenerate_section']['auto_regenerate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-regenerate when content changes'),
      '#description' => $this->t('Automatically update llms.txt files when pages or menus are modified. The regeneration happens in the background via a queue system.'),
      '#default_value' => $config->get('auto_regenerate') ?? TRUE,
    ];

    $form['auto_regenerate_section']['debounce_interval'] = [
      '#type' => 'number',
      '#title' => $this->t('Debouncing interval (minutes)'),
      '#description' => $this->t('Minimum time between auto-regenerations. Lower values = more reactive but higher API costs. Higher values = less reactive but lower API costs.'),
      '#default_value' => $config->get('debounce_interval') ?? 5,
      '#min' => 1,
      '#max' => 60,
      '#step' => 1,
      '#states' => [
        'visible' => [
          ':input[name="auto_regenerate"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * AJAX callback to reformulate summary.
   */
  public function reformulateSummaryAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    try {
      $current_text = $form_state->getValue('summary') ?? '';
      $site_name = $this->config('system.site')->get('name');
      $site_slogan = $this->config('system.site')->get('slogan') ?? '';

      $suggestion = $this->introductionReformulation->generateSummary(
        $current_text,
        $site_name,
        $site_slogan
      );

      // Update the textarea value using data-field-type selector.
      // KISS: same pattern as DescriptionsManagementForm.
      $selector = 'textarea[data-field-type="summary"]';
      $response->addCommand(new InvokeCommand($selector, 'val', [$suggestion]));

      $this->messenger()->addStatus($this->t('AI summary generated. Please review and modify as needed before saving.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to generate AI summary: @message', ['@message' => $e->getMessage()]));
    }

    return $response;
  }

  /**
   * AJAX callback to reformulate introduction.
   */
  public function reformulateIntroductionAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    try {
      $current_text = $form_state->getValue('introduction') ?? '';
      $site_name = $this->config('system.site')->get('name');
      $site_slogan = $this->config('system.site')->get('slogan') ?? '';

      $suggestion = $this->introductionReformulation->generateIntroduction(
        $current_text,
        $site_name,
        $site_slogan
      );

      // Update the textarea value using data-field-type selector.
      // KISS: same pattern as DescriptionsManagementForm.
      $selector = 'textarea[data-field-type="introduction"]';
      $response->addCommand(new InvokeCommand($selector, 'val', [$suggestion]));

      $this->messenger()->addStatus($this->t('AI introduction generated. Please review and modify as needed before saving.'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Failed to generate AI introduction: @message', ['@message' => $e->getMessage()]));
    }

    return $response;
  }

  /**
   * AJAX callback to import homepage meta description for summary.
   */
  public function importFromHomepageSummaryAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    // Only process if this button was actually clicked via AJAX.
    $triggering_element = $form_state->getTriggeringElement();
    if (empty($triggering_element) || $triggering_element['#name'] !== 'import_summary_button') {
      // Not our button, return empty response without any messages.
      return $response;
    }

    try {
      $homepage_meta = $this->introductionReformulation->getHomepageMetaDescription();

      if (empty($homepage_meta)) {
        // Use MessageCommand to display error without persisting to session.
        $response->addCommand(new MessageCommand(
          $this->t('No homepage meta description found. Make sure your homepage is a node with a meta description.'),
          NULL,
          ['type' => 'error']
        ));
      }
      else {
        // Update the textarea value with the raw meta description.
        $response->addCommand(new InvokeCommand('textarea[data-field-type="summary"]', 'val', [$homepage_meta]));
        // Use MessageCommand to display success without persisting to session.
        $response->addCommand(new MessageCommand(
          $this->t('Homepage description imported. You can now edit or reformulate it.'),
          NULL,
          ['type' => 'status']
        ));
      }
    }
    catch (\Exception $e) {
      // Use MessageCommand to display error without persisting to session.
      $response->addCommand(new MessageCommand(
        $this->t('Failed to import homepage description: @message', ['@message' => $e->getMessage()]),
        NULL,
        ['type' => 'error']
      ));
    }

    return $response;
  }

  /**
   * AJAX callback to import homepage meta description for introduction.
   */
  public function importFromHomepageIntroductionAjax(array &$form, FormStateInterface $form_state) {
    $response = new AjaxResponse();

    // Only process if this button was actually clicked.
    $triggering_element = $form_state->getTriggeringElement();
    if (empty($triggering_element) || $triggering_element['#name'] !== 'import_introduction_button') {
      return $response;
    }

    try {
      $homepage_meta = $this->introductionReformulation->getHomepageMetaDescription();

      if (empty($homepage_meta)) {
        // Use MessageCommand to display error without persisting to session.
        $response->addCommand(new MessageCommand(
          $this->t('No homepage meta description found. Make sure your homepage is a node with a meta description.'),
          NULL,
          ['type' => 'error']
        ));
      }
      else {
        // Update the textarea value with the raw meta description.
        $response->addCommand(new InvokeCommand('textarea[data-field-type="introduction"]', 'val', [$homepage_meta]));
        // Use MessageCommand to display success without persisting to session.
        $response->addCommand(new MessageCommand(
          $this->t('Homepage description imported. You can now edit or reformulate it.'),
          NULL,
          ['type' => 'status']
        ));
      }
    }
    catch (\Exception $e) {
      // Use MessageCommand to display error without persisting to session.
      $response->addCommand(new MessageCommand(
        $this->t('Failed to import homepage description: @message', ['@message' => $e->getMessage()]),
        NULL,
        ['type' => 'error']
      ));
    }

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('llms_txt_ai.settings')
      ->set('summary', $form_state->getValue('summary'))
      ->set('introduction', $form_state->getValue('introduction'))
      ->set('menus', array_filter($form_state->getValue('menus')))
      ->set('menu_depth', $form_state->getValue('menu_depth'))
      ->set('include_homepage', $form_state->getValue('include_homepage'))
      ->set('auto_regenerate', $form_state->getValue('auto_regenerate'))
      ->set('debounce_interval', $form_state->getValue('debounce_interval'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Gets available menu options.
   *
   * @return array
   *   Array of menu options.
   */
  protected function getMenuOptions(): array {
    $menus = $this->entityTypeManager->getStorage('menu')->loadMultiple();
    $options = [];
    foreach ($menus as $menu_id => $menu) {
      $options[$menu_id] = $menu->label();
    }
    return $options;
  }

}
