<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;

/**
 * Service to store manual description overrides.
 */
class DescriptionStorageService {

  /**
   * The key-value store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface
   */
  protected $storage;

  /**
   * Constructs a DescriptionStorageService object.
   *
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $key_value_factory
   *   The key-value factory.
   */
  public function __construct(KeyValueFactoryInterface $key_value_factory) {
    $this->storage = $key_value_factory->get('llms_txt_ai_manual_descriptions');
  }

  /**
   * Sets a manual description for a node.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string $description
   *   The description text.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   */
  public function set(int|string $nid, string $description, ?string $langcode = NULL): void {
    $key = $this->buildKey($nid, $langcode);
    $this->storage->set($key, $description);
  }

  /**
   * Gets a manual description for a node.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return string|null
   *   The description or NULL if not set.
   */
  public function get(int|string $nid, ?string $langcode = NULL): ?string {
    $key = $this->buildKey($nid, $langcode);
    return $this->storage->get($key);
  }

  /**
   * Gets all manual descriptions.
   *
   * @return array
   *   Array of descriptions keyed by NID.
   */
  public function getAll(): array {
    return $this->storage->getAll();
  }

  /**
   * Deletes a manual description.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   */
  public function delete(int|string $nid, ?string $langcode = NULL): void {
    $key = $this->buildKey($nid, $langcode);
    $this->storage->delete($key);
  }

  /**
   * Builds a storage key with language support.
   *
   * @param int|string $nid
   *   The node ID or special identifier.
   * @param string|null $langcode
   *   The language code. If NULL, returns key without language suffix.
   *
   * @return string
   *   The storage key.
   */
  protected function buildKey(int|string $nid, ?string $langcode = NULL): string {
    $key = (string) $nid;
    if ($langcode !== NULL) {
      $key .= ':' . $langcode;
    }
    return $key;
  }

}
