<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service to generate the final llms.txt file.
 */
class LlmsTxtGeneratorService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The description storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\DescriptionStorageService
   */
  protected $descriptionStorage;

  /**
   * The reformulated storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\ReformulatedStorageService
   */
  protected $reformulatedStorage;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The AI reformulation service.
   *
   * @var \Drupal\llms_txt_ai\Service\AiReformulationService
   */
  protected $aiReformulation;

  /**
   * Constructs a LlmsTxtGeneratorService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\llms_txt_ai\Service\MenuExtractorService $menu_extractor
   *   The menu extractor service.
   * @param \Drupal\llms_txt_ai\Service\DescriptionStorageService $description_storage
   *   The description storage service.
   * @param \Drupal\llms_txt_ai\Service\ReformulatedStorageService $reformulated_storage
   *   The reformulated storage service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $url_generator
   *   The URL generator.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\llms_txt_ai\Service\AiReformulationService $ai_reformulation
   *   The AI reformulation service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    MenuExtractorService $menu_extractor,
    DescriptionStorageService $description_storage,
    ReformulatedStorageService $reformulated_storage,
    CacheBackendInterface $cache,
    UrlGeneratorInterface $url_generator,
    StateInterface $state,
    RequestStack $request_stack,
    LanguageManagerInterface $language_manager,
    AiReformulationService $ai_reformulation,
  ) {
    $this->configFactory = $config_factory;
    $this->menuExtractor = $menu_extractor;
    $this->descriptionStorage = $description_storage;
    $this->reformulatedStorage = $reformulated_storage;
    $this->cache = $cache;
    $this->urlGenerator = $url_generator;
    $this->state = $state;
    $this->requestStack = $request_stack;
    $this->languageManager = $language_manager;
    $this->aiReformulation = $ai_reformulation;
  }

  /**
   * Generates the llms.txt content for the default language.
   *
   * @return string
   *   The llms.txt content.
   */
  public function generate(): string {
    $default_langcode = $this->languageManager->getDefaultLanguage()->getId();
    return $this->generateForLanguage($default_langcode);
  }

  /**
   * Generates the llms.txt content for a specific language.
   *
   * @param string $langcode
   *   The language code.
   *
   * @return string
   *   The llms.txt content.
   */
  public function generateForLanguage(string $langcode): string {
    $config = $this->configFactory->get('llms_txt_ai.settings');

    // Get configuration.
    $summary_source = $config->get('summary') ?? '';
    $introduction_source = $config->get('introduction') ?? '';
    $menus = $config->get('menus') ?? [];
    $include_homepage = $config->get('include_homepage') ?? TRUE;

    // AI always translates to target language (KISS principle).
    // The AI prompts contain translation instructions, so it will
    // automatically translate if source is in a different language.
    $summary = $summary_source;
    $introduction = $introduction_source;

    if (!empty($summary_source)) {
      try {
        $summary = $this->aiReformulation->reformulate($summary_source, 'summary', $langcode);
      }
      catch (\Exception $e) {
        // Fallback to original if AI fails.
        $summary = $summary_source;
      }
    }

    if (!empty($introduction_source)) {
      try {
        $introduction = $this->aiReformulation->reformulate($introduction_source, 'introduction', $langcode);
      }
      catch (\Exception $e) {
        // Fallback to original if AI fails.
        $introduction = $introduction_source;
      }
    }

    // Build content.
    $content = [];
    $base_url = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();

    // Site name (H1).
    $site_name = $this->configFactory->get('system.site')->get('name');
    $content[] = "# {$site_name}";
    $content[] = '';

    // Summary (blockquote).
    if (!empty($summary)) {
      $content[] = "> {$summary}";
      $content[] = '';
    }

    // Introduction (optional details).
    if (!empty($introduction)) {
      $content[] = $introduction;
      $content[] = '';
    }

    // Collect all NIDs for cleanup.
    $all_nids = [];

    // Homepage (before sections if exists).
    if ($include_homepage) {
      $homepage = $this->menuExtractor->getHomepage($langcode);
      if ($homepage) {
        $nid = $homepage['nid'];
        $title_source = $homepage['title'];
        $url = $base_url . $homepage['url'];

        // AI always translates title to target language (KISS principle).
        $title = $title_source;
        if (!empty($title_source)) {
          try {
            $title = $this->aiReformulation->reformulate($title_source, 'title', $langcode);
          }
          catch (\Exception $e) {
            // Fallback to original if AI fails.
            $title = $title_source;
          }
        }

        $description = $this->getDescription($nid, $title_source, $langcode);

        $content[] = "- [{$title}]({$url}): {$description}";
        $content[] = '';
        $all_nids[] = $nid;
      }
    }

    // Extract pages grouped by menu.
    $menus_data = $this->menuExtractor->extractPagesGroupedByMenu($menus, $langcode);

    // Generate sections (H2) for each menu.
    foreach ($menus_data as $menu_data) {
      $menu_name_source = $menu_data['menu_name'];
      $pages = $menu_data['pages'];

      if (empty($pages)) {
        continue;
      }

      // AI always translates menu name to target language (KISS principle).
      $menu_name = $menu_name_source;
      if (!empty($menu_name_source)) {
        try {
          $menu_name = $this->aiReformulation->reformulate($menu_name_source, 'title', $langcode);
        }
        catch (\Exception $e) {
          // Fallback to original if AI fails.
          $menu_name = $menu_name_source;
        }
      }

      $content[] = "## {$menu_name}";
      $content[] = '';

      foreach ($pages as $page) {
        $nid = $page['nid'];
        $title_source = $page['title'];
        $url = $base_url . $page['url'];

        // AI always translates title to target language (KISS principle).
        $title = $title_source;
        if (!empty($title_source)) {
          try {
            $title = $this->aiReformulation->reformulate($title_source, 'title', $langcode);
          }
          catch (\Exception $e) {
            // Fallback to original if AI fails.
            $title = $title_source;
          }
        }

        $description = $this->getDescription($nid, $title_source, $langcode);

        $content[] = "- [{$title}]({$url}): {$description}";
        $all_nids[] = $nid;
      }

      $content[] = '';
    }

    $content[] = '---';
    $content[] = 'Generated by llms.txt AI Generator for Drupal';

    $output = implode("\n", $content);

    // Store in State API with language suffix.
    $state_key = 'llms_txt_ai.generated_content.' . $langcode;
    $this->state->set($state_key, $output);
    $this->state->set('llms_txt_ai.generated_timestamp.' . $langcode, time());

    // Also cache for performance.
    $cache_key = 'llms_txt_ai:output:' . $langcode;
    $this->cache->set($cache_key, $output, CacheBackendInterface::CACHE_PERMANENT, ['llms_txt_ai_output']);

    // Cleanup orphaned entries for this language.
    $this->reformulatedStorage->cleanup($all_nids, $langcode);

    return $output;
  }

  /**
   * Gets the description for a page.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string $fallback_title
   *   Fallback title if no description found.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return string
   *   The description.
   */
  protected function getDescription(int|string $nid, string $fallback_title, ?string $langcode = NULL): string {
    // Priority 1: Manual override.
    $manual = $this->descriptionStorage->get($nid, $langcode);
    if (!empty($manual)) {
      return $manual;
    }

    // Priority 2: Reformulated description.
    $reformulated_data = $this->reformulatedStorage->get($nid, $langcode);
    if (!empty($reformulated_data['reformulated'])) {
      return $reformulated_data['reformulated'];
    }

    // Priority 3: Fallback to title.
    return $fallback_title;
  }

  /**
   * Gets llms.txt content for a specific language.
   *
   * First tries cache for performance, then falls back to State API.
   * State API persists across cache clears.
   *
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return string|null
   *   Content or NULL if not generated.
   */
  public function getCached(?string $langcode = NULL): ?string {
    // Use default language if not specified.
    if ($langcode === NULL) {
      $langcode = $this->languageManager->getDefaultLanguage()->getId();
    }

    // Try cache first for performance.
    $cache_key = 'llms_txt_ai:output:' . $langcode;
    $cached = $this->cache->get($cache_key);
    if ($cached) {
      return $cached->data;
    }

    // Fall back to State API (survives cache clears).
    $state_key = 'llms_txt_ai.generated_content.' . $langcode;
    $content = $this->state->get($state_key);

    // If found in State, restore to cache for next time.
    if (!empty($content)) {
      $this->cache->set($cache_key, $content, CacheBackendInterface::CACHE_PERMANENT, ['llms_txt_ai_output']);
      return $content;
    }

    return NULL;
  }

  /**
   * Deletes the generated content for all languages.
   */
  public function invalidateCache(): void {
    // Get all enabled languages.
    $languages = $this->languageManager->getLanguages();

    foreach ($languages as $language) {
      $langcode = $language->getId();
      $this->cache->delete('llms_txt_ai:output:' . $langcode);
      $this->state->delete('llms_txt_ai.generated_content.' . $langcode);
      $this->state->delete('llms_txt_ai.generated_timestamp.' . $langcode);
    }

    // Also delete old format for backward compatibility.
    $this->cache->delete('llms_txt_ai:output');
    $this->state->delete('llms_txt_ai.generated_content');
    $this->state->delete('llms_txt_ai.generated_timestamp');
  }

  /**
   * Gets the timestamp of last generation for a language.
   *
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return int|null
   *   Unix timestamp or NULL if never generated.
   */
  public function getGeneratedTimestamp(?string $langcode = NULL): ?int {
    if ($langcode === NULL) {
      $langcode = $this->languageManager->getDefaultLanguage()->getId();
    }
    return $this->state->get('llms_txt_ai.generated_timestamp.' . $langcode);
  }

  /**
   * Gets all enabled language codes.
   *
   * @return array
   *   Array of language codes.
   */
  public function getEnabledLanguages(): array {
    $languages = $this->languageManager->getLanguages();
    return array_keys($languages);
  }

}
