<?php

namespace Drupal\llms_txt_ai\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Controller for serving llms.txt file.
 */
class LlmsTxtController extends ControllerBase {

  /**
   * The generator service.
   *
   * @var \Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService
   */
  protected $generator;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected $urlGenerator;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs a LlmsTxtController object.
   *
   * @param \Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService $generator
   *   The generator service.
   * @param \Drupal\Core\Routing\UrlGeneratorInterface $url_generator
   *   The URL generator.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(
    LlmsTxtGeneratorService $generator,
    UrlGeneratorInterface $url_generator,
    LanguageManagerInterface $language_manager,
  ) {
    $this->generator = $generator;
    $this->urlGenerator = $url_generator;
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get("llms_txt_ai.generator"),
      $container->get("url_generator"),
      $container->get("language_manager"),
    );
  }

  /**
   * Serves the llms.txt file.
   *
   * Uses the current language from Drupal's language negotiation system.
   * This allows URLs like /fr/llms.txt to automatically serve French content.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function output() {
    // Get current language from Drupal's language negotiation.
    // This is set automatically based on URL prefix (/fr/, /en/, etc.).
    $current_language = $this->languageManager
      ->getCurrentLanguage()
      ->getId();

    // Try to get cached content for the current language.
    $content = $this->generator->getCached($current_language);

    if (empty($content)) {
      throw new NotFoundHttpException(
        "llms.txt file not generated for language '{$current_language}'. Please generate it from the administration interface at /admin/config/content/llms-txt-ai/generate",
      );
    }

    $response = new Response($content);
    $response->headers->set("Content-Type", "text/plain; charset=utf-8");
    $response->headers->set("Cache-Control", "public, max-age=3600");

    return $response;
  }

  /**
   * Displays the llms.txt file in the admin interface.
   *
   * @return array
   *   A render array.
   */
  public function view() {
    // Try to get cached content.
    $content = $this->generator->getCached();

    if (empty($content)) {
      $build = [
        "#type" => "markup",
        "#markup" =>
        '<div class="messages messages--warning">' .
        $this->t(
            'The llms.txt file has not been generated yet. Please go to the <a href="@generate">Generate tab</a> to create it.',
            [
              "@generate" =>
              "/admin/config/content/llms-txt-ai/generate",
            ],
        ) .
        "</div>",
      ];
    }
    else {
      // Display the content in a formatted way.
      $build = [
        "#type" => "container",
        "#attributes" => ["class" => ["llms-txt-viewer"]],
      ];

      // Add info message with public URL.
      $public_url = $this->urlGenerator->generateFromRoute(
        "llms_txt_ai.output",
        [],
        ["absolute" => TRUE],
      );
      $build["info"] = [
        "#type" => "markup",
        "#markup" =>
        '<div class="messages messages--status">' .
        $this->t(
            'Your llms.txt file is publicly accessible at: <a href="@url" target="_blank">@url</a>',
            [
              "@url" => $public_url,
            ],
        ) .
        "</div>",
      ];

      // Add the content in a pre-formatted block.
      $build["content"] = [
        "#type" => "markup",
        "#markup" =>
        '<pre style="background: #f5f5f5; padding: 20px; border: 1px solid #ddd; border-radius: 4px; overflow-x: auto; font-family: monospace; line-height: 1.6;">' .
        htmlspecialchars($content) .
        "</pre>",
      ];

      // Add copy button.
      $build["actions"] = [
        "#type" => "markup",
        "#markup" =>
        '<div style="margin-top: 20px;">' .
        '<a href="' .
        $public_url .
        '" target="_blank" class="button">' .
        $this->t("🔗 Open Public URL") .
        "</a>" .
        "</div>",
      ];
    }

    return $build;
  }

}
