<?php

namespace Drupal\llms_txt_ai\Plugin\QueueWorker;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\State\StateInterface;
use Drupal\llms_txt_ai\Service\AiReformulationService;
use Drupal\llms_txt_ai\Service\ChangeDetectionService;
use Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService;
use Drupal\llms_txt_ai\Service\MenuExtractorService;
use Drupal\llms_txt_ai\Service\MetaDescriptionService;
use Drupal\llms_txt_ai\Service\ReformulatedStorageService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Queue worker for auto-regenerating llms.txt files.
 *
 * @QueueWorker(
 *   id = "llms_txt_ai_regenerate",
 *   title = @Translation("llms.txt Auto-Regenerate"),
 *   cron = {"time" = 60}
 * )
 */
class LlmsTxtRegenerateQueueWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The llms.txt generator service.
   *
   * @var \Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService
   */
  protected $generator;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The change detection service.
   *
   * @var \Drupal\llms_txt_ai\Service\ChangeDetectionService
   */
  protected $changeDetection;

  /**
   * The AI reformulation service.
   *
   * @var \Drupal\llms_txt_ai\Service\AiReformulationService
   */
  protected $aiReformulation;

  /**
   * The reformulated storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\ReformulatedStorageService
   */
  protected $reformulatedStorage;

  /**
   * The meta description service.
   *
   * @var \Drupal\llms_txt_ai\Service\MetaDescriptionService
   */
  protected $metaDescription;

  /**
   * Flag to prevent recursion during regeneration.
   *
   * @var bool
   */
  private static $regenerating = FALSE;

  /**
   * Constructs a new LlmsTxtRegenerateQueueWorker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService $generator
   *   The llms.txt generator service.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\llms_txt_ai\Service\MenuExtractorService $menu_extractor
   *   The menu extractor service.
   * @param \Drupal\llms_txt_ai\Service\ChangeDetectionService $change_detection
   *   The change detection service.
   * @param \Drupal\llms_txt_ai\Service\AiReformulationService $ai_reformulation
   *   The AI reformulation service.
   * @param \Drupal\llms_txt_ai\Service\ReformulatedStorageService $reformulated_storage
   *   The reformulated storage service.
   * @param \Drupal\llms_txt_ai\Service\MetaDescriptionService $meta_description
   *   The meta description service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LlmsTxtGeneratorService $generator,
    StateInterface $state,
    LoggerInterface $logger,
    ConfigFactoryInterface $config_factory,
    MenuExtractorService $menu_extractor,
    ChangeDetectionService $change_detection,
    AiReformulationService $ai_reformulation,
    ReformulatedStorageService $reformulated_storage,
    MetaDescriptionService $meta_description,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->generator = $generator;
    $this->state = $state;
    $this->logger = $logger;
    $this->configFactory = $config_factory;
    $this->menuExtractor = $menu_extractor;
    $this->changeDetection = $change_detection;
    $this->aiReformulation = $ai_reformulation;
    $this->reformulatedStorage = $reformulated_storage;
    $this->metaDescription = $meta_description;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('llms_txt_ai.generator'),
      $container->get('state'),
      $container->get('logger.factory')->get('llms_txt_ai'),
      $container->get('config.factory'),
      $container->get('llms_txt_ai.menu_extractor'),
      $container->get('llms_txt_ai.change_detection'),
      $container->get('llms_txt_ai.ai_reformulation'),
      $container->get('llms_txt_ai.reformulated_storage'),
      $container->get('llms_txt_ai.meta_description')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data) {
    // Prevent recursion during regeneration.
    if (self::$regenerating) {
      return;
    }

    $langcode = $data['langcode'] ?? NULL;
    if (!$langcode) {
      $this->logger->error('Auto-regenerate queue item missing langcode');
      return;
    }

    // Check if recently generated (debouncing).
    if ($this->isRecentlyGenerated($langcode)) {
      $this->logger->debug('Skipping auto-regeneration for @lang (recently generated)', ['@lang' => $langcode]);
      return;
    }

    try {
      // Set flag to prevent recursion.
      self::$regenerating = TRUE;

      // Get configured menus.
      $config = $this->configFactory->get('llms_txt_ai.settings');
      $menus = array_filter($config->get('menus') ?? []);

      if (empty($menus)) {
        $this->logger->warning('No menus configured for auto-regeneration');
        return;
      }

      // Extract pages for this language.
      $pages = $this->menuExtractor->extractPages($menus, $langcode);

      // Detect changes.
      $changes = $this->changeDetection->detectChanges($pages, $langcode);

      // Reformulate changed pages.
      $reformulated_count = 0;
      foreach ($changes['to_reformulate'] as $nid) {
        $page = $this->findPageByNid($pages, $nid);
        if (!$page) {
          $this->logger->warning('Page NID @nid not found in pages array for language @lang', [
            '@nid' => $nid,
            '@lang' => $langcode,
          ]);
          continue;
        }

        // Get source description.
        $source_text = $this->getSourceDescription($page, $langcode);

        // Reformulate with AI.
        $reformulated = $this->aiReformulation->reformulate($source_text, 'description', $langcode);

        // Store reformulated data.
        $this->reformulatedStorage->set($nid, [
          'source_text' => $source_text,
          'source_hash' => md5($source_text),
          'reformulated' => $reformulated,
          'timestamp' => time(),
          'ai_provider' => $config->get('ai_provider'),
        ], $langcode);

        $reformulated_count++;
      }

      // Generate the final file.
      $this->generator->generateForLanguage($langcode);

      // Update last auto-regeneration timestamp.
      $this->state->set("llms_txt_ai.last_auto_regen.$langcode", time());

      $this->logger->info('Auto-regenerated llms.txt for language @lang: @count pages reformulated', [
        '@lang' => $langcode,
        '@count' => $reformulated_count,
      ]);

    }
    catch (\Exception $e) {
      // Log error but don't re-throw to avoid infinite retries.
      $this->logger->error('Auto-regeneration failed for @lang: @error', [
        '@lang' => $langcode,
        '@error' => $e->getMessage(),
      ]);
    }
    finally {
      // Always reset the flag.
      self::$regenerating = FALSE;
    }
  }

  /**
   * Checks if llms.txt was recently generated for a language.
   *
   * Implements debouncing to avoid over-regeneration.
   *
   * @param string $langcode
   *   The language code.
   *
   * @return bool
   *   TRUE if recently generated, FALSE otherwise.
   */
  protected function isRecentlyGenerated(string $langcode): bool {
    $config = $this->configFactory->get('llms_txt_ai.settings');
    $debounce_minutes = $config->get('debounce_interval') ?? 5;
    // Convert minutes to seconds.
    $debounce_interval = $debounce_minutes * 60;

    $last_generated = $this->state->get("llms_txt_ai.last_auto_regen.$langcode", 0);
    return (time() - $last_generated) < $debounce_interval;
  }

  /**
   * Gets source description for a page.
   *
   * @param array $page
   *   Page array with keys: title, url, nid.
   * @param string|null $langcode
   *   The language code. If NULL, uses current language.
   *
   * @return string
   *   Source description (meta or title).
   */
  protected function getSourceDescription(array $page, ?string $langcode = NULL): string {
    $meta = $this->metaDescription->getMetaDescription($page['nid'], $langcode);
    return !empty($meta) ? $meta : $page['title'];
  }

  /**
   * Finds a page by NID in the pages array.
   *
   * @param array $pages
   *   Array of pages.
   * @param int $nid
   *   Node ID to find.
   *
   * @return array|null
   *   Page array or NULL if not found.
   */
  protected function findPageByNid(array $pages, int $nid): ?array {
    foreach ($pages as $page) {
      // Use loose comparison to handle string/int type differences.
      if ($page['nid'] == $nid) {
        return $page;
      }
    }
    return NULL;
  }

}
