<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service to reformulate text using AI.
 */
class AiReformulationService {

  /**
   * Default prompt for summary reformulation.
   */
  const DEFAULT_PROMPT_SUMMARY = <<<'EOT'
<context>
You are helping create content for an llms.txt file - a standardized format that helps AI assistants understand websites.
The summary is the first thing AI assistants read to quickly understand what this website is about.
</context>

<language_requirement>
You MUST write the summary in {language}. If the source text is in a different language, translate it to {language} first, then reformulate it according to the instructions below.
</language_requirement>

<instructions>
Write a concise, informative summary that:
- Is exactly 1-2 sentences (maximum 50 words)
- Clearly states what the website does or offers
- Uses simple, direct language
- Focuses on the core value proposition
- Avoids marketing jargon and buzzwords
- Is written in third person
- Is written in {language}
</instructions>

<constraints>
- Maximum length: 2 sentences or 50 words
- No promotional language ("best", "leading", "revolutionary")
- No vague terms ("innovative", "cutting-edge", "world-class")
- No calls to action
- Focus on facts, not opinions
- MUST be in {language}
</constraints>

<input_text>
{text}
</input_text>

<output_format>
Return ONLY the rewritten summary text in {language}, without any preamble, explanation, or meta-commentary.
</output_format>
EOT;

  /**
   * Default prompt for introduction reformulation.
   */
  const DEFAULT_PROMPT_INTRODUCTION = <<<'EOT'
<context>
You are helping create content for an llms.txt file - a standardized format that helps AI assistants understand websites.
The introduction provides a comprehensive overview that helps AI assistants deeply understand the website's purpose, audience, and value.
</context>

<language_requirement>
You MUST write the introduction in {language}. If the source text is in a different language, translate it to {language} first, then reformulate it according to the instructions below.
</language_requirement>

<instructions>
Rewrite this introduction to be clear, informative, and well-structured:
- Write 2-3 concise paragraphs (150-250 words total)
- First paragraph: What the website is and its primary purpose
- Second paragraph: Who it's for and what problems it solves
- Third paragraph (optional): Key features, approach, or unique aspects
- Use natural, conversational language
- Be specific and concrete, not vague or abstract
- Focus on information that helps AI assistants understand context
- Write in {language}
</instructions>

<constraints>
- Length: 2-3 paragraphs, 150-250 words total
- No marketing hyperbole or promotional language
- No bullet points or lists (use flowing paragraphs)
- No calls to action or sales pitches
- Avoid jargon unless necessary (then explain it)
- Write in third person
- MUST be in {language}
</constraints>

<tone>
Professional yet approachable, informative without being dry, clear without being simplistic.
</tone>

<input_text>
{text}
</input_text>

<output_format>
Return ONLY the rewritten introduction text in {language} in paragraph form, without any preamble, explanation, or meta-commentary.
</output_format>
EOT;

  /**
   * Default prompt for description reformulation.
   */
  const DEFAULT_PROMPT_DESCRIPTION = <<<'EOT'
<context>
You are helping create content for an llms.txt file - a standardized format that helps AI assistants understand websites.
This description is derived from SEO meta descriptions and should provide a clear, concise explanation of a specific page or section.
</context>

<language_requirement>
You MUST write the description in {language}. If the source text is in a different language, translate it to {language} first, then reformulate it according to the instructions below.
</language_requirement>

<instructions>
Transform this SEO meta description into natural, AI-friendly language:
- Keep it concise: 1-2 sentences maximum
- Remove SEO-optimized keywords stuffing
- Make it conversational and easy to understand
- Focus on what the page actually contains or does
- Preserve the core information while improving clarity
- Use active voice when possible
- Write in {language}
</instructions>

<constraints>
- Maximum length: 2 sentences or 40 words
- No keyword stuffing or SEO tricks
- No promotional language or superlatives
- No calls to action ("Learn more", "Contact us", etc.)
- No special characters or emojis
- MUST be in {language}
</constraints>

<examples>
BAD: "Discover our amazing products! Best prices guaranteed. Shop now for exclusive deals on top-quality items. Free shipping available!"
GOOD: "An online store offering electronics, home goods, and accessories with free shipping on orders over $50."

BAD: "Leading innovative solutions for enterprise-level digital transformation and cutting-edge technology implementation."
GOOD: "Provides software consulting services to help businesses modernize their technology infrastructure."
</examples>

<input_text>
{text}
</input_text>

<output_format>
Return ONLY the rewritten description text in {language}, without any preamble, explanation, or meta-commentary.
</output_format>
EOT;

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * The prompt manager service.
   *
   * @var \Drupal\llms_txt_ai\Service\PromptManagerService
   */
  protected $promptManager;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Constructs an AiReformulationService object.
   *
   * @param \Drupal\ai\AiProviderPluginManager $ai_provider_manager
   *   The AI provider manager.
   * @param \Drupal\llms_txt_ai\Service\PromptManagerService $prompt_manager
   *   The prompt manager service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   */
  public function __construct(
    AiProviderPluginManager $ai_provider_manager,
    PromptManagerService $prompt_manager,
    LoggerChannelFactoryInterface $logger_factory,
    LanguageManagerInterface $language_manager,
  ) {
    $this->aiProviderManager = $ai_provider_manager;
    $this->promptManager = $prompt_manager;
    $this->logger = $logger_factory->get('llms_txt_ai');
    $this->languageManager = $language_manager;
  }

  /**
   * Reformulates text using AI.
   *
   * @param string $text
   *   The text to reformulate.
   * @param string $context
   *   Context type: 'description' or 'introduction'.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return string
   *   The reformulated text, or original text if AI fails.
   */
  public function reformulate(string $text, string $context = 'description', ?string $langcode = NULL): string {
    try {
      // Get the default provider and model for chat operation.
      $operation_type = 'chat';
      $default = $this->aiProviderManager->getDefaultProviderForOperationType($operation_type);

      if (empty($default['provider_id']) || empty($default['model_id'])) {
        $this->logger->error('No default provider or model set for chat operation. Please configure AI settings at /admin/config/ai.');
        return $text;
      }

      // Create provider instance.
      $provider = $this->aiProviderManager->createInstance($default['provider_id']);
      $model_id = $default['model_id'];

      // Build prompt based on context and language.
      $prompt = $this->buildPrompt($text, $context, $langcode);

      // Create ChatInput with ChatMessage.
      $message = new ChatMessage('user', $prompt);
      $input = new ChatInput([$message]);

      // Call AI provider with proper ChatInput object and model ID.
      $output = $provider->chat($input, $model_id);

      // Extract normalized text from ChatOutput.
      if (!empty($output)) {
        $normalized = $output->getNormalized();
        if ($normalized) {
          return trim($normalized->getText());
        }
      }

      return $text;
    }
    catch (\Exception $e) {
      $this->logger->error('AI reformulation failed: @message', ['@message' => $e->getMessage()]);
      return $text;
    }
  }

  /**
   * Builds the AI prompt based on context.
   *
   * @param string $text
   *   The text to reformulate.
   * @param string $context
   *   Context type: 'description', 'introduction', or 'summary'.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return string
   *   The prompt for the AI.
   */
  protected function buildPrompt(string $text, string $context, ?string $langcode = NULL): string {
    // Get the prompt content from the prompt manager.
    // This will use user prompt if available, otherwise system prompt.
    $prompt_template = $this->promptManager->getPromptContent($context);

    // Fallback to default constants if no prompt found.
    if (empty($prompt_template)) {
      if ($context === 'summary') {
        $prompt_template = self::DEFAULT_PROMPT_SUMMARY;
      }
      elseif ($context === 'introduction') {
        $prompt_template = self::DEFAULT_PROMPT_INTRODUCTION;
      }
      else {
        $prompt_template = self::DEFAULT_PROMPT_DESCRIPTION;
      }
    }

    // Determine language name for {language} placeholder.
    // Use explicit English names to ensure AI understands the language.
    $language_name = $this->getLanguageEnglishName($langcode);

    // Replace placeholders with actual values.
    $replacements = [
      '{text}' => $text,
      '{language}' => $language_name,
    ];
    $prompt = str_replace(array_keys($replacements), array_values($replacements), $prompt_template);

    // Debug logging to verify prompt construction.
    $this->logger->debug('AI Prompt built - Context: @context, Langcode: @langcode, Language: @language, Text length: @length', [
      '@context' => $context,
      '@langcode' => $langcode ?? 'NULL',
      '@language' => $language_name,
      '@length' => strlen($text),
    ]);

    return $prompt;
  }

  /**
   * Gets the English name of a language from its langcode.
   *
   * Uses a manual mapping to ensure consistent English names for AI prompts,
   * regardless of the interface language.
   *
   * @param string|null $langcode
   *   The language code (e.g., 'fr', 'en', 'de').
   *
   * @return string
   *   The English name of the language (e.g., 'French', 'English', 'German').
   */
  protected function getLanguageEnglishName(?string $langcode): string {
    // Manual mapping of common language codes to English names.
    // This ensures AI always receives consistent English language names.
    $language_map = [
      'en' => 'English',
      'fr' => 'French',
      'de' => 'German',
      'es' => 'Spanish',
      'it' => 'Italian',
      'pt' => 'Portuguese',
      'pt-br' => 'Brazilian Portuguese',
      'nl' => 'Dutch',
      'pl' => 'Polish',
      'ru' => 'Russian',
      'ja' => 'Japanese',
      'zh-hans' => 'Simplified Chinese',
      'zh-hant' => 'Traditional Chinese',
      'ar' => 'Arabic',
      'he' => 'Hebrew',
      'hi' => 'Hindi',
      'ko' => 'Korean',
      'sv' => 'Swedish',
      'da' => 'Danish',
      'no' => 'Norwegian',
      'fi' => 'Finnish',
      'cs' => 'Czech',
      'hu' => 'Hungarian',
      'ro' => 'Romanian',
      'tr' => 'Turkish',
      'el' => 'Greek',
      'uk' => 'Ukrainian',
    ];

    // Return mapped name if available, otherwise default to English.
    if ($langcode !== NULL && isset($language_map[$langcode])) {
      return $language_map[$langcode];
    }

    // Fallback: try to get language name from Drupal.
    if ($langcode !== NULL) {
      $language = $this->languageManager->getLanguage($langcode);
      if ($language) {
        // Use label() which returns the English name in most cases.
        return $language->getName();
      }
    }

    // Ultimate fallback.
    return 'English';
  }

}
