<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\menu_link_content\Entity\MenuLinkContent;
use Drupal\node\NodeInterface;

/**
 * Service for managing automatic regeneration of llms.txt files.
 */
class AutoRegenerateService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The queue factory.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queueFactory;

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an AutoRegenerateService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Queue\QueueFactory $queue_factory
   *   The queue factory.
   * @param \Drupal\llms_txt_ai\Service\MenuExtractorService $menu_extractor
   *   The menu extractor service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory, LanguageManagerInterface $language_manager, QueueFactory $queue_factory, MenuExtractorService $menu_extractor, $logger_factory) {
    $this->configFactory = $config_factory;
    $this->languageManager = $language_manager;
    $this->queueFactory = $queue_factory;
    $this->menuExtractor = $menu_extractor;
    $this->logger = $logger_factory->get('llms_txt_ai');
  }

  /**
   * Checks if auto-regeneration is enabled.
   *
   * @return bool
   *   TRUE if auto-regeneration is enabled, FALSE otherwise.
   */
  public function isAutoRegenerateEnabled(): bool {
    $config = $this->configFactory->get('llms_txt_ai.settings');
    return $config->get('auto_regenerate') ?? TRUE;
  }

  /**
   * Handles entity changes that might require regeneration.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity that was changed.
   * @param string $operation
   *   The operation performed (insert, update, delete).
   */
  public function handleEntityChange(EntityInterface $entity, string $operation): void {
    // Skip if auto-regeneration is disabled.
    if (!$this->isAutoRegenerateEnabled()) {
      return;
    }

    // Only handle nodes and menu links.
    if (!($entity instanceof NodeInterface) && !($entity instanceof MenuLinkContent)) {
      return;
    }

    // For nodes, check if they're in configured menus.
    if ($entity instanceof NodeInterface && !$this->isNodeRelevant($entity)) {
      return;
    }

    // For menu links, check if they're in configured menus.
    if ($entity instanceof MenuLinkContent && !$this->isMenuLinkRelevant($entity)) {
      return;
    }

    // Queue regeneration for all active languages.
    $this->queueRegenerationForAllLanguages();

    $this->logger->debug('Queued auto-regeneration due to @type @operation: @label', [
      '@type' => $entity->getEntityTypeId(),
      '@operation' => $operation,
      '@label' => $entity->label(),
    ]);
  }

  /**
   * Checks if a node is relevant for llms.txt generation.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to check.
   *
   * @return bool
   *   TRUE if the node is relevant, FALSE otherwise.
   */
  protected function isNodeRelevant(NodeInterface $node): bool {
    // Only published nodes are relevant.
    if (!$node->isPublished()) {
      return FALSE;
    }

    // Check if node is accessible to anonymous users.
    if (!$node->access('view')) {
      return FALSE;
    }

    // Check if node is in any of the configured menus.
    $config = $this->configFactory->get('llms_txt_ai.settings');
    $configured_menus = array_filter($config->get('menus') ?? []);
    if (empty($configured_menus)) {
      return FALSE;
    }

    // This is a simplified check - in practice, we'd need to check
    // if the node appears in any of the configured menus.
    // For now, we assume any published, accessible node might be relevant.
    return TRUE;
  }

  /**
   * Checks if a menu link is relevant for llms.txt generation.
   *
   * @param \Drupal\menu_link_content\Entity\MenuLinkContent $menu_link
   *   The menu link to check.
   *
   * @return bool
   *   TRUE if the menu link is relevant, FALSE otherwise.
   */
  protected function isMenuLinkRelevant(MenuLinkContent $menu_link): bool {
    // Check if the menu link is in one of the configured menus.
    $config = $this->configFactory->get('llms_txt_ai.settings');
    $configured_menus = array_filter($config->get('menus') ?? []);

    $menu_name = $menu_link->getMenuName();
    return in_array($menu_name, $configured_menus, TRUE);
  }

  /**
   * Queues regeneration for all active languages.
   */
  protected function queueRegenerationForAllLanguages(): void {
    $queue = $this->queueFactory->get('llms_txt_ai_regenerate');

    foreach ($this->languageManager->getLanguages() as $language) {
      $langcode = $language->getId();
      // Add item to queue (will be deduplicated if already exists).
      $queue->createItem(['langcode' => $langcode]);
    }
  }

}
