<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service to handle batch processing for AI reformulation.
 */
class BatchService {

  use StringTranslationTrait;

  /**
   * The AI reformulation service.
   *
   * @var \Drupal\llms_txt_ai\Service\AiReformulationService
   */
  protected $aiReformulation;

  /**
   * The reformulated storage service.
   *
   * @var \Drupal\llms_txt_ai\Service\ReformulatedStorageService
   */
  protected $reformulatedStorage;

  /**
   * The generator service.
   *
   * @var \Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService
   */
  protected $generator;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a BatchService object.
   *
   * @param \Drupal\llms_txt_ai\Service\AiReformulationService $ai_reformulation
   *   The AI reformulation service.
   * @param \Drupal\llms_txt_ai\Service\ReformulatedStorageService $reformulated_storage
   *   The reformulated storage service.
   * @param \Drupal\llms_txt_ai\Service\LlmsTxtGeneratorService $generator
   *   The generator service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    AiReformulationService $ai_reformulation,
    ReformulatedStorageService $reformulated_storage,
    LlmsTxtGeneratorService $generator,
    ConfigFactoryInterface $config_factory,
  ) {
    $this->aiReformulation = $ai_reformulation;
    $this->reformulatedStorage = $reformulated_storage;
    $this->generator = $generator;
    $this->configFactory = $config_factory;
  }

  /**
   * Creates a batch array for processing pages.
   *
   * @param array $pages_data
   *   Array of pages with their source descriptions and langcode.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return array
   *   Batch array.
   */
  public function createBatch(array $pages_data, ?string $langcode = NULL): array {
    $operations = [];

    // Split pages into chunks of 10.
    $chunks = array_chunk($pages_data, 10, TRUE);

    foreach ($chunks as $chunk) {
      $operations[] = [
        [static::class, 'processChunk'],
        [$chunk, $langcode],
      ];
    }

    // Final operation: generate the file.
    $operations[] = [
      [static::class, 'generateFinalFile'],
      [$langcode],
    ];

    return [
      'title' => $this->t('Generating llms.txt...'),
      'operations' => $operations,
      'finished' => [static::class, 'finishBatch'],
      'progress_message' => $this->t('Processing @current of @total pages...'),
    ];
  }

  /**
   * Batch operation: Process a chunk of pages.
   *
   * @param array $chunk
   *   Chunk of pages to process.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   * @param array &$context
   *   Batch context.
   */
  public static function processChunk(array $chunk, ?string $langcode, array &$context): void {
    $ai_reformulation = \Drupal::service('llms_txt_ai.ai_reformulation');
    $reformulated_storage = \Drupal::service('llms_txt_ai.reformulated_storage');
    $config = \Drupal::config('llms_txt_ai.settings');

    if (!isset($context['results']['processed'])) {
      $context['results']['processed'] = 0;
    }

    foreach ($chunk as $nid => $data) {
      $source_text = $data['source_text'];
      $page_langcode = $data['langcode'] ?? $langcode;

      // Reformulate with AI.
      $reformulated = $ai_reformulation->reformulate($source_text, 'description', $page_langcode);

      // Calculate hash.
      $source_hash = md5($source_text);

      // Store reformulated data.
      $reformulated_storage->set($nid, [
        'source_text' => $source_text,
        'source_hash' => $source_hash,
        'reformulated' => $reformulated,
        'timestamp' => time(),
        'ai_provider' => $config->get('ai_provider'),
      ], $page_langcode);

      $context['results']['processed']++;
      $lang_suffix = $page_langcode ? " ({$page_langcode})" : '';
      $context['message'] = t('Processed @count pages@lang...', [
        '@count' => $context['results']['processed'],
        '@lang' => $lang_suffix,
      ]);
    }
  }

  /**
   * Batch operation: Generate the final llms.txt file.
   *
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   * @param array &$context
   *   Batch context.
   */
  public static function generateFinalFile(?string $langcode, array &$context): void {
    $generator = \Drupal::service('llms_txt_ai.generator');

    // Generate and cache the file.
    $generator->generateForLanguage($langcode ?? \Drupal::languageManager()->getDefaultLanguage()->getId());

    $lang_suffix = $langcode ? " ({$langcode})" : '';
    $context['message'] = t('Generating final llms.txt file@lang...', ['@lang' => $lang_suffix]);
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Whether the batch completed successfully.
   * @param array $results
   *   Batch results.
   * @param array $operations
   *   Remaining operations.
   */
  public static function finishBatch(bool $success, array $results, array $operations): void {
    if ($success) {
      $processed = $results['processed'] ?? 0;
      \Drupal::messenger()->addStatus(t('Successfully generated llms.txt with @count reformulated pages.', ['@count' => $processed]));
      \Drupal::messenger()->addStatus(t('View your file at <a href="@url" target="_blank">@url</a>', [
        '@url' => \Drupal::request()->getSchemeAndHttpHost() . '/llms.txt',
      ]));
    }
    else {
      \Drupal::messenger()->addError(t('An error occurred during generation.'));
    }
  }

}
