<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service to help generate/reformulate introduction text.
 */
class IntroductionReformulationService {

  /**
   * The AI reformulation service.
   *
   * @var \Drupal\llms_txt_ai\Service\AiReformulationService
   */
  protected $aiReformulation;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The menu extractor service.
   *
   * @var \Drupal\llms_txt_ai\Service\MenuExtractorService
   */
  protected $menuExtractor;

  /**
   * The meta description service.
   *
   * @var \Drupal\llms_txt_ai\Service\MetaDescriptionService
   */
  protected $metaDescription;

  /**
   * Constructs an IntroductionReformulationService object.
   *
   * @param \Drupal\llms_txt_ai\Service\AiReformulationService $ai_reformulation
   *   The AI reformulation service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\llms_txt_ai\Service\MenuExtractorService $menu_extractor
   *   The menu extractor service.
   * @param \Drupal\llms_txt_ai\Service\MetaDescriptionService $meta_description
   *   The meta description service.
   */
  public function __construct(
    AiReformulationService $ai_reformulation,
    ConfigFactoryInterface $config_factory,
    MenuExtractorService $menu_extractor,
    MetaDescriptionService $meta_description,
  ) {
    $this->aiReformulation = $ai_reformulation;
    $this->configFactory = $config_factory;
    $this->menuExtractor = $menu_extractor;
    $this->metaDescription = $meta_description;
  }

  /**
   * Gets the homepage meta description.
   *
   * @return string
   *   The homepage meta description or empty string if not found.
   */
  public function getHomepageMetaDescription(): string {
    $homepage = $this->menuExtractor->getHomepage();
    if ($homepage && isset($homepage['nid'])) {
      return $this->metaDescription->getMetaDescription($homepage['nid']) ?? '';
    }
    return '';
  }

  /**
   * Generates or improves summary text using AI.
   *
   * @param string $current_text
   *   Current summary text (can be empty).
   * @param string $site_name
   *   The site name.
   * @param string $site_slogan
   *   The site slogan (optional).
   *
   * @return string
   *   The AI-generated summary suggestion.
   */
  public function generateSummary(string $current_text, string $site_name, string $site_slogan = ''): string {
    if (!empty($current_text)) {
      // Improve existing text.
      return $this->aiReformulation->reformulate($current_text, 'summary');
    }

    // Try to get homepage meta description.
    $homepage_meta = $this->getHomepageMetaDescription();

    if (!empty($homepage_meta)) {
      // Reformulate the homepage meta description.
      return $this->aiReformulation->reformulate($homepage_meta, 'summary');
    }

    // Generate from scratch.
    $prompt = "Write a very short, clear summary (1-2 sentences maximum) for a website called '{$site_name}'";
    if (!empty($site_slogan)) {
      $prompt .= " with the slogan '{$site_slogan}'";
    }
    $prompt .= ". Be concise and engaging.";

    return $this->aiReformulation->reformulate($prompt, 'summary');
  }

  /**
   * Generates or improves introduction text using AI.
   *
   * @param string $current_text
   *   Current introduction text (can be empty).
   * @param string $site_name
   *   The site name.
   * @param string $site_slogan
   *   The site slogan (optional).
   *
   * @return string
   *   The AI-generated introduction suggestion.
   */
  public function generateIntroduction(string $current_text, string $site_name, string $site_slogan = ''): string {
    if (!empty($current_text)) {
      // Improve existing text.
      return $this->aiReformulation->reformulate($current_text, 'introduction');
    }

    // Try to get homepage meta description.
    $homepage_meta = $this->getHomepageMetaDescription();

    if (!empty($homepage_meta)) {
      // Reformulate the homepage meta description.
      return $this->aiReformulation->reformulate($homepage_meta, 'introduction');
    }

    // Generate from scratch.
    $prompt = "Write a clear, engaging introduction (2-3 paragraphs) for a website called '{$site_name}'";
    if (!empty($site_slogan)) {
      $prompt .= " with the slogan '{$site_slogan}'";
    }
    $prompt .= ". Explain what the site does and why it's useful. Write in natural, conversational language suitable for AI assistants.";

    return $this->aiReformulation->reformulate($prompt, 'introduction');
  }

}
