<?php

namespace Drupal\llms_txt_ai\Service;

use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;

/**
 * Service to store AI-reformulated descriptions.
 */
class ReformulatedStorageService {

  /**
   * The key-value store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface
   */
  protected $storage;

  /**
   * Constructs a ReformulatedStorageService object.
   *
   * @param \Drupal\Core\KeyValueStore\KeyValueFactoryInterface $key_value_factory
   *   The key-value factory.
   */
  public function __construct(KeyValueFactoryInterface $key_value_factory) {
    $this->storage = $key_value_factory->get('llms_txt_ai_reformulated');
  }

  /**
   * Sets a reformulated description for a node.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param array $data
   *   Array with keys: source_text, source_hash, reformulated,
   *   timestamp, ai_provider.
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   */
  public function set(int|string $nid, array $data, ?string $langcode = NULL): void {
    $key = $this->buildKey($nid, $langcode);
    $this->storage->set($key, $data);
  }

  /**
   * Gets a reformulated description for a node.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   *
   * @return array|null
   *   The reformulated data or NULL if not set.
   */
  public function get(int|string $nid, ?string $langcode = NULL): ?array {
    $key = $this->buildKey($nid, $langcode);
    return $this->storage->get($key);
  }

  /**
   * Gets all reformulated descriptions.
   *
   * @return array
   *   Array of reformulated data keyed by NID.
   */
  public function getAll(): array {
    return $this->storage->getAll();
  }

  /**
   * Deletes a reformulated description.
   *
   * @param int|string $nid
   *   The node ID or special identifier (e.g., 'homepage_manual').
   * @param string|null $langcode
   *   The language code. If NULL, uses default language.
   */
  public function delete(int|string $nid, ?string $langcode = NULL): void {
    $key = $this->buildKey($nid, $langcode);
    $this->storage->delete($key);
  }

  /**
   * Cleans up orphaned entries (pages no longer in menus).
   *
   * @param array $current_nids
   *   Array of current NIDs that should be kept.
   * @param string|null $langcode
   *   The language code. If NULL, cleans all languages.
   */
  public function cleanup(array $current_nids, ?string $langcode = NULL): void {
    $all_data = $this->getAll();
    foreach (array_keys($all_data) as $key) {
      // Extract NID from key (format: "nid" or "nid:langcode").
      $parts = explode(':', $key);
      $nid = (int) $parts[0];
      $key_langcode = $parts[1] ?? NULL;

      // If langcode specified, only clean that language.
      if ($langcode !== NULL && $key_langcode !== $langcode) {
        continue;
      }

      if (!in_array($nid, $current_nids)) {
        $this->storage->delete($key);
      }
    }
  }

  /**
   * Builds a storage key with language support.
   *
   * @param int|string $nid
   *   The node ID or special identifier.
   * @param string|null $langcode
   *   The language code. If NULL, returns key without language suffix.
   *
   * @return string
   *   The storage key.
   */
  protected function buildKey(int|string $nid, ?string $langcode = NULL): string {
    $key = (string) $nid;
    if ($langcode !== NULL) {
      $key .= ':' . $langcode;
    }
    return $key;
  }

}
