<?php

declare(strict_types=1);

namespace Drupal\lms_answer_plugins\Plugin\ActivityAnswer;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Component\Utility\Html;
use Drupal\lms\Attribute\ActivityAnswer;
use Drupal\lms\Entity\Answer;
use Drupal\lms\Plugin\ActivityAnswerBase;

/**
 * Free Text with Feedback activity plugin.
 */
#[ActivityAnswer(
  id: 'free_text_feedback',
  name: new TranslatableMarkup('Free text with feedback'),
)]
final class FreeTextFeedback extends ActivityAnswerBase {

  /**
   * {@inheritdoc}
   */
  public function evaluatedOnSave(Answer $activity): bool {
    // Answer is automatically evaluated by phrase matching.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getScore(Answer $answer): float {
    $activity = $answer->getActivity();
    $data = $answer->getData();

    // If no answer has been provided yet, return 0.
    if (!\array_key_exists('answer', $data) || $data['answer'] === '') {
      return 0.0;
    }

    $student_answer = $data['answer'];
    $phrases_feedback = $activity->get('phrase_feedback');

    // If no phrases defined, give full score for any answer.
    if ($phrases_feedback->count() === 0) {
      return 1.0;
    }

    // Count how many phrases match.
    $matched_phrases = 0;
    $total_phrases = \count($phrases_feedback);

    foreach ($phrases_feedback as $item) {
      if ($this->isPhraseFound($student_answer, $item->get('phrase')->getString())) {
        $matched_phrases++;
      }
    }

    // Calculate score based on percentage of matched phrases.
    return $total_phrases > 0 ? (float) ($matched_phrases / $total_phrases) : 1.0;
  }

  /**
   * {@inheritdoc}
   */
  public function answeringForm(array &$form, FormStateInterface $form_state, Answer $answer): void {
    $activity = $answer->getActivity();
    $data = $answer->getData();
    $activity_id = $activity->id();
    $activity_selector = 'activity-' . $activity_id;

    // Add the answer textarea.
    $form['answer'] = [
      '#title' => $this->t('Your answer'),
      '#type' => 'textarea',
      '#default_value' => array_key_exists('answer', $data) ? $data['answer'] : '',
      '#rows' => 10,
      '#required' => TRUE,
      '#element_validate' => [[$this, 'validateAnswer']],
    ];

    // Add feedback container with Activity ID for one-page lessons.
    $form['feedback'] = [
      '#type' => 'container',
      '#attributes' => [
        'data-lms-selector' => 'feedback-' . $activity_id,
      ],
      '#weight' => 10,
    ];

    // Hide submit button on initial display if no answer yet.
    if (
      $form_state->getValue('answer') === NULL &&
      !\array_key_exists('answer', $answer->getData())
    ) {
      $form['actions']['submit']['#access'] = FALSE;
    }

    // Add 'Check Answer' button before Submit button.
    $form['actions']['check'] = [
      '#type' => 'button',
      '#value' => $this->t('Check Answer'),
      '#ajax' => [
        'callback' => [$this, 'getFeedback'],
      ],
      '#weight' => 1,
      '#attributes' => [
        'data-lms-selector' => $activity_selector,
      ],
    ];

    $form['actions']['submit']['#weight'] = 2;
    $form['#attributes']['data-lms-selector'] = $activity_selector;
  }

  /**
   * Element validate callback for minimum character validation.
   */
  public function validateAnswer(array $element, FormStateInterface $form_state): void {
    $form_object = $form_state->getFormObject();
    assert($form_object instanceof ContentEntityFormInterface);
    $answer = $form_object->getEntity();
    assert($answer instanceof Answer);
    $activity = $answer->getActivity();

    // Skip validation if there is no minimum character requirement.
    $min_chars_field = $activity->get('minimum_characters');
    $minimum_characters = (int) $min_chars_field->first()->getValue()['value'];
    if ($min_chars_field->isEmpty() || $minimum_characters <= 0) {
      return;
    }

    // Calculate the character count after stripping tags and line breaks.
    $value = \strip_tags($form_state->getValue('answer'));
    $value = \preg_replace('/\r\n?/', '', $value);
    $total = \mb_strlen($value);

    if ($total < $minimum_characters) {
      $args = [
        '@min_message' => $this->formatPlural($minimum_characters,
          'We are looking for a minimum of @count character,',
          'We are looking for a minimum of @count characters,'),
        '@total_message' => $this->formatPlural($total,
          'and your answer is @count character.',
          'and your answer is @count characters.'),
      ];
      $form_state->setError($element, $this->t('@min_message @total_message', $args));
    }
  }

  /**
   * Feedback callback.
   */
  public function getFeedback(array $form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    $form_object = $form_state->getFormObject();
    assert($form_object instanceof ContentEntityFormInterface);
    $answer = $form_object->getEntity();
    assert($answer instanceof Answer);

    $current_answer = $form_state->getValue('answer');
    $answer->setData(['answer' => $current_answer]);

    $activity = $answer->getActivity();
    $activity_id = $activity->id();

    // Replace actions - submit button is now visible.
    $response->addCommand(new ReplaceCommand('.form-actions', $form['actions']));

    // Evaluate each phrase match and build feedback.
    $has_matched = FALSE;
    $has_unmatched = FALSE;
    foreach ($activity->get('phrase_feedback') as $delta => $item) {
      $phrase = $item->get('phrase')->getString();

      // Check whether student's answer contains phrase match.
      if ($this->isPhraseFound($current_answer, $phrase)) {
        $feedback_text = $item->get('feedback_present')->getString();
        if ($feedback_text !== '') {

          // Create matched feedback container for first matched item.
          if (!$has_matched) {
            $form['feedback']['content']['matched'] = [
              '#type' => 'html_tag',
              '#tag' => 'div',
              '#attributes' => ['class' => ['feedback-section', 'matched-phrases']],
              'heading' => [
                '#type' => 'html_tag',
                '#tag' => 'h4',
                '#value' => $this->t('You seem to have a good understanding of:'),
                '#weight' => -10,
              ],
            ];
            $has_matched = TRUE;
          }

          // Add phrase matching feedback text.
          $form['feedback']['content']['matched']['items'][$delta] = [
            '#type' => 'html_tag',
            '#tag' => 'p',
            '#attributes' => ['class' => ['phrase-match', 'correct-answer']],
            '#value' => $feedback_text,
          ];
        }
      }
      else {
        $feedback_text = $item->get('feedback_absent')->getString();
        if ($feedback_text !== '') {

          // Create unmatched feedback container for first unmatched item.
          if (!$has_unmatched) {
            $form['feedback']['content']['unmatched'] = [
              '#type' => 'html_tag',
              '#tag' => 'div',
              '#attributes' => ['class' => ['feedback-section', 'unmatched-phrases']],
              'heading' => [
                '#type' => 'html_tag',
                '#tag' => 'h4',
                '#value' => $this->t('You could have another look at:'),
                '#weight' => -10,
              ],
            ];
            $has_unmatched = TRUE;
          }

          // Add non-matching feedback text.
          $form['feedback']['content']['unmatched']['items'][$delta] = [
            '#type' => 'html_tag',
            '#tag' => 'p',
            '#attributes' => ['class' => ['phrase-mismatch', 'wrong-answer']],
            '#value' => $feedback_text,
          ];
        }
      }
    }

    $response->addCommand(new ReplaceCommand(
      '[data-lms-selector="feedback-' . $activity_id . '"]',
      $form['feedback']
    ));

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function evaluationDisplay(Answer $answer): array {
    $data = $answer->getData();

    $answer_markup = '';
    if (array_key_exists('answer', $data)) {
      $answer_markup = \nl2br(Html::escape($data['answer']));
    }

    return [
      // Render activity.
      'activity' => $this->entityTypeManager->getViewBuilder('lms_activity')->view($answer->getActivity(), 'activity'),
      // Add answer.
      'answer' => [
        '#type' => 'fieldset',
        '#title' => $this->t('Student answer'),
        'answer' => [
          '#markup' => $answer_markup,
        ],
      ],
    ];
  }

  /**
   * Checks if a phrase is found in the student's answer.
   *
   * @param string $student_answer
   *   The student's answer text.
   * @param string $phrase_with_variants
   *   Phrase with possible variants separated by |.
   *
   * @return bool
   *   TRUE if the phrase or any of its variants is found.
   */
  private function isPhraseFound(string $student_answer, string $phrase_with_variants): bool {
    $phrase_variants = \explode('|', Html::escape($phrase_with_variants));

    foreach ($phrase_variants as $phrase) {
      $phrase = \trim($phrase);
      if ($phrase === '') {
        continue;
      }

      // Case-insensitive substring match.
      if (\stripos($student_answer, $phrase) !== FALSE) {
        return TRUE;
      }
    }

    return FALSE;
  }

}
