<?php

declare(strict_types=1);

namespace Drupal\lms;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Security\Attribute\TrustedCallback;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\lms\Entity\ActivityInterface;
use Drupal\lms\Entity\Bundle\Course;
use Drupal\lms\Entity\CourseStatusInterface;
use Drupal\lms\Entity\LessonInterface;
use Drupal\lms\Entity\LessonStatusInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;

/**
 * Builds the course navigation block.
 */
class BlockBuilder {

  use StringTranslationTrait;

  private const CACHE_PREFIX = 'steps_block_data_cache_';

  /**
   * Lesson IDs of this course.
   */
  private ?array $lessonIds = NULL;

  /**
   * Instance cache for user's lesson statuses.
   */
  private ?array $userLessonStatuses = NULL;

  /**
   * Instance cache for route parameters.
   */
  private ?array $routeParameters = NULL;

  /**
   * Instance cache for user's course status.
   */
  private ?CourseStatusInterface $userCourseStatus = NULL;

  /**
   * Static cache for course structure to reduce DB lookups across requests.
   */
  private array $courseStructureCache = [];

  /**
   * {@inheritdoc}
   */
  public function __construct(
    #[Autowire(service: 'cache.default')]
    protected readonly CacheBackendInterface $cacheBackend,
    protected readonly TrainingManager $trainingManager,
    protected readonly RouteMatchInterface $currentRoute,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * Get route parameter.
   */
  private function getRouteParameter(string $parameter_name): ?int {
    if ($this->routeParameters === NULL) {
      $this->routeParameters = [];
      foreach (['lesson_delta', 'activity_delta'] as $parameter_key) {
        $parameter_value = $this->currentRoute->getParameter($parameter_key);
        $this->routeParameters[$parameter_key] = $parameter_value !== NULL ? (int) $parameter_value : NULL;
      }
    }
    return $this->routeParameters[$parameter_name] ?? NULL;
  }

  /**
   * Set initial object parameters for lesson status getter.
   *
   * @return bool
   *   Do we have all required data?
   */
  private function initialize(Course $course, AccountInterface $account, array $lesson_ids): bool {
    // Already initialized.
    if ($this->lessonIds !== NULL) {
      return TRUE;
    }
    $this->userCourseStatus = $this->trainingManager->loadCourseStatus($course, $account, ['current' => TRUE]);
    if ($this->userCourseStatus === NULL) {
      return FALSE;
    }
    $this->lessonIds = $lesson_ids;
    return TRUE;
  }

  /**
   * Lesson status getter.
   */
  private function getLessonStatus(string $lesson_id): ?LessonStatusInterface {
    // Set static cache.
    if ($this->userLessonStatuses === NULL) {
      $this->userLessonStatuses = [];

      $statuses = $this->trainingManager->loadLessonStatusMultiple($this->userCourseStatus->id(), $this->lessonIds);
      foreach ($statuses as $status) {
        $lesson_target_id = $status->get('lesson')->target_id;
        if ($lesson_target_id !== '') {
          $this->userLessonStatuses[$lesson_target_id] = $status;
        }
      }
    }

    return \array_key_exists($lesson_id, $this->userLessonStatuses) ? $this->userLessonStatuses[$lesson_id] : NULL;
  }

  /**
   * Retrieves the ordered lesson structure for a course, with caching.
   */
  public function getStructure(Course $group): array {
    $course_id = $group->id();
    if ($course_id === NULL) {
      return [];
    }

    // Check static cache for course structure first.
    if (\array_key_exists($course_id, $this->courseStructureCache)) {
      return $this->courseStructureCache[$course_id];
    }

    $cache_id = self::CACHE_PREFIX . 'structure_' . $course_id;
    $cache_item = $this->cacheBackend->get($cache_id);
    if ($cache_item !== FALSE) {
      $this->courseStructureCache[$course_id] = $cache_item->data;
      return $cache_item->data;
    }

    // If not in the cache, build the structure.
    $structure = $this->trainingManager->getOrderedLessons($group);
    $tags = ['group:' . $course_id];
    foreach (\array_keys($structure) as $lesson_id) {
      $tags[] = 'lms_lesson:' . $lesson_id;
    }

    $this->cacheBackend->set($cache_id, $structure, Cache::PERMANENT, $tags);
    $this->courseStructureCache[$course_id] = $structure;
    return $structure;
  }

  /**
   * Build the main course navigation block.
   */
  public function build(Course $course, AccountInterface $account): array {
    $build = [
      '#type' => 'component',
      '#component' => 'lms:course_navigation',
      '#props' => [
        'title' => $course->label(),
      ],
      '#slots' => [
        'lessons' => [],
      ],
    ];

    $structure = $this->getStructure($course);
    if (!$this->initialize($course, $account, \array_keys($structure))) {
      return $build;
    }

    $course_id = $course->id();
    $user_id = $account->id();

    $current_lesson_route_delta = $this->getRouteParameter('lesson_delta');

    // Initialize course progress calculation and pre-load lesson statuses.
    $total_activities = 0;
    $answered_activities = 0;

    $overall_cache_meta = new CacheableMetadata();
    $overall_cache_meta->addCacheContexts([
      'user',
      'url.path',
      'url.query_args:lesson_delta',
      'url.query_args:activity_delta',
    ]);
    if ($course_id !== NULL) {
      $overall_cache_meta->addCacheTags(['group:' . $course_id]);
    }
    if ($user_id >= 0) {
      $overall_cache_meta->addCacheTags(['user:' . $user_id]);
    }
    $overall_cache_meta->setCacheMaxAge(Cache::PERMANENT);

    // Build all lessons with appropriate cache settings.
    foreach ($structure as $lesson_id => $lesson_data) {
      $is_current_lesson = ($current_lesson_route_delta === $lesson_data['delta']);

      $lesson_cache_tags = [
        'group:' . $course_id,
        'lms_lesson:' . $lesson_id,
      ];

      // Adjust course progress to include this lesson.
      if (\array_key_exists('activities', $lesson_data) && \is_array($lesson_data['activities'])) {
        $lesson_total = \count($lesson_data['activities']);
        $total_activities += $lesson_total;

        // Count answered activities in this lesson.
        $lesson_status = $this->getLessonStatus((string) $lesson_id);
        if ($lesson_status !== NULL) {
          $lesson_cache_tags[] = 'lms_lesson_status:' . $lesson_status->id();
          $activity_ids = \array_map('intval', \array_keys($lesson_data['activities']));
          $answers = $this->trainingManager->getAnswersByActivityId($lesson_status, $activity_ids);
          $answered_activities += \count($answers);
        }
      }

      // Add this lesson to the render array.
      $build['#slots']['lessons'][$lesson_id] = [
        '#lazy_builder' => [
          static::class . '::buildLesson',
          [$lesson_id, $is_current_lesson, $course_id, $user_id, $lesson_data['delta']],
        ],
        '#create_placeholder' => TRUE,
        '#cache' => [
          'keys' => ['lms-lesson', $lesson_id, $course_id],
          'contexts' => ['user', 'url.query_args'],
          'tags' => $lesson_cache_tags,
        ],
      ];
      $overall_cache_meta->addCacheTags($lesson_cache_tags);
    }

    $overall_cache_meta->applyTo($build);

    // Calculate overall course progress percentage.
    $progress_percentage = $total_activities > 0 ?
      \round(($answered_activities / $total_activities) * 100) : 0;

    // Update component props with course progress.
    $build['#props']['progress_percentage'] = $progress_percentage;
    $build['#props']['progress_text'] = $this->t('@answered of @total total answered.', [
      '@answered' => $answered_activities,
      '@total' => $total_activities,
    ]);

    return $build;
  }

  /**
   * Lesson lazy builder callback actual callback.
   */
  public function doBuildLesson(
    string $lesson_id,
    bool $is_on_route,
    string $course_id,
    string $user_id,
    int $lesson_route_delta,
  ): array {
    $course = $this->entityTypeManager->getStorage('group')->load($course_id);
    $account = $this->entityTypeManager->getStorage('user')->load($user_id);
    $lesson = $this->entityTypeManager->getStorage('lms_lesson')->load($lesson_id);
    if (
      !$lesson instanceof LessonInterface ||
      !$course instanceof Course ||
      !$account instanceof AccountInterface
    ) {
      return [];
    }

    $structure = $this->getStructure($course);
    if (!\array_key_exists($lesson_id, $structure)) {
      return [];
    }
    if (!$this->initialize($course, $account, \array_keys($structure))) {
      return [];
    }

    $this->trainingManager->loadCourseStatus($course, $account, ['current' => TRUE]);
    $show_scores = $this->userCourseStatus->isFinished() &&
      $course->revisitMode();

    $lesson_structure_data = $structure[$lesson_id];
    $current_activity_route_delta = $this->getRouteParameter('activity_delta');

    $cacheableMetadata = new CacheableMetadata();
    $cacheableMetadata->addCacheableDependency($this->userCourseStatus);
    $cacheableMetadata->addCacheableDependency($lesson);

    // Build lesson properties.
    $lesson_props = [
      'title' => $lesson_structure_data['label'],
      'url' => Url::fromRoute('lms.group.answer_form', [
        'group' => $course->id(),
        'lesson_delta' => $lesson_structure_data['delta'],
      ])->toString(),
      'is_current' => $is_on_route,
      'started' => FALSE,
      'finished' => FALSE,
      'evaluated' => FALSE,
      'passed' => FALSE,
    ];

    // Get current course position.
    $current_lesson_delta = NULL;
    $current_activity_delta = 0;

    $current_lesson_status = $this->userCourseStatus->getCurrentLessonStatus();
    if ($current_lesson_status instanceof LessonStatusInterface) {
      $current_lesson_delta = $current_lesson_status->getCurrentLessonDelta();
      $current_activity_delta = $current_lesson_status->getCurrentActivityDelta() ?? 0;
    }
    else {
      $current_lesson_delta = 0;
    }

    // Update lesson properties based on status.
    $lesson_status = $this->getLessonStatus($lesson_id);
    if ($lesson_status instanceof LessonStatusInterface) {
      $lesson_props['started'] = TRUE;
      $lesson_props['evaluated'] = $lesson_status->isEvaluated();
      if ($lesson_status->isFinished()) {
        $lesson_props['finished'] = TRUE;
        $lesson_props['passed'] = $lesson_status->getScore() >= $lesson_status->getRequiredScore();
      }
      $cacheableMetadata->addCacheableDependency($lesson_status);
    }

    // Build activities.
    $activities_build_slot = ['#type' => 'container', '#attributes' => ['class' => ['lms-activities-list']]];
    $activities_from_source = [];
    $answers_for_lesson = [];

    // Get activities from lesson status or course structure.
    if ($lesson_status instanceof LessonStatusInterface) {
      $activities_field = $lesson_status->get(LessonInterface::ACTIVITIES);
      $activity_ids_for_query = [];
      foreach ($activities_field as $index => $item) {
        \assert($item instanceof EntityReferenceItem);
        if ($item->entity instanceof ActivityInterface && $item->target_id !== NULL) {
          $activities_from_source[(string) $item->target_id] = ['entity' => $item->entity, 'delta' => $index];
          $activity_ids_for_query[] = (int) $item->target_id;
        }
      }
      $answers_for_lesson = \count($activity_ids_for_query) > 0 ?
        $this->trainingManager->getAnswersByActivityId($lesson_status, $activity_ids_for_query) : [];
    }
    // Get from course structure only if lesson randomization is off.
    elseif ($lesson->getRandomization() === 0) {
      if (\array_key_exists('activities', $lesson_structure_data) && \is_array($lesson_structure_data['activities'])) {
        foreach ($lesson_structure_data['activities'] as $activity_id_from_structure => $activity_data_from_structure) {
          if ($activity_data_from_structure['activity'] instanceof ActivityInterface) {
            $activities_from_source[(string) $activity_id_from_structure] = [
              'entity' => $activity_data_from_structure['activity'],
              'delta' => $activity_data_from_structure['delta'],
            ];
          }
        }
      }
    }

    // Process each activity.
    foreach ($activities_from_source as $activity_id => $activity_info) {
      $activity = $activity_info['entity'];
      $activity_route_delta = (int) $activity_info['delta'];
      $is_current_activity = $is_on_route && $activity_route_delta === $current_activity_route_delta;
      $is_answered = \array_key_exists($activity_id, $answers_for_lesson);
      $answer = $is_answered ? $answers_for_lesson[$activity_id] : NULL;
      $is_evaluated = $is_answered && $answer->isEvaluated();
      $current_score = $is_answered ? $answer->getScore() : NULL;
      $max_score = $this->trainingManager->getActivityMaxScore($lesson, $activity);
      $max_score_achieved = ($max_score > 0 && $current_score !== NULL && $current_score >= $max_score);

      // Determine URL access.
      $url_enabled = FALSE;

      $deltas_for_check = [
        'lesson' => $lesson_route_delta,
        'activity' => $activity_route_delta,
        'current_lesson' => $current_lesson_delta,
        'current_activity' => $current_activity_delta,
      ];
      $access_code = $this->trainingManager->checkActivityAccess($this->userCourseStatus, $deltas_for_check);
      $url_enabled = ($access_code === NULL);
      if ($is_current_activity) {
        $url_enabled = FALSE;
      }

      $activities_build_slot[$activity_id] = [
        '#type' => 'component',
        '#component' => 'lms:activity_item',
        '#props' => [
          'title' => $activity->label(),
          'url' => $url_enabled ? Url::fromRoute('lms.group.answer_form', [
            'group' => $course->id(),
            'lesson_delta' => $lesson_route_delta,
            'activity_delta' => $activity_route_delta,
          ])->toString() : '',
          'url_enabled' => $url_enabled,
          'is_current' => $is_current_activity,
          'answered' => $is_answered,
          'evaluated' => $is_evaluated,
          'max_score_achieved' => $show_scores ? $max_score_achieved : FALSE,
          'score' => $show_scores ? $current_score : NULL,
          'max_score' => $show_scores ? $max_score : NULL,
          'show_score' => $show_scores,
        ],
      ];

      if ($is_answered) {
        $cacheableMetadata->addCacheableDependency($answer);
      }
    }

    $build = [
      '#type' => 'component',
      '#component' => 'lms:lesson_item',
      '#props' => $lesson_props,
      '#slots' => ['activities' => $activities_build_slot],
    ];

    $cacheableMetadata->applyTo($build);
    return ['lesson' => $build];
  }

  /**
   * Lazy builder callback to render a lesson, populated with its activities.
   *
   * @param string $lesson_id
   *   The lesson ID.
   * @param bool $is_on_route
   *   Whether this is the current lesson.
   * @param string $course_id
   *   The course ID.
   * @param string $user_id
   *   The user ID.
   * @param int $lesson_route_delta
   *   The lesson route delta.
   *
   * @return array
   *   The render array.
   */
  #[TrustedCallback]
  public static function buildLesson(
    string $lesson_id,
    bool $is_on_route,
    string $course_id,
    string $user_id,
    int $lesson_route_delta,
  ): array {
    return \Drupal::service(self::class)->doBuildLesson($lesson_id, $is_on_route, $course_id, $user_id, $lesson_route_delta);
  }

}
