<?php

declare(strict_types=1);

namespace Drupal\lms_answer_plugins\Plugin\ActivityAnswer;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lms\Attribute\ActivityAnswer;
use Drupal\lms\Entity\ActivityInterface;
use Drupal\lms\Entity\Answer;
use Drupal\lms_answer_plugins\Plugin\SelectBase;

/**
 * Select Multiple with Feedback activity plugin.
 */
#[ActivityAnswer(
  id: 'select_multiple_feedback',
  name: new TranslatableMarkup('Multiple answer select with feedback'),
)]
final class SelectMultipleFeedback extends SelectBase {

  use \Drupal\lms_answer_plugins\Plugin\WithFeedbackPluginTrait;

  protected const ELEMENT_TYPE = 'checkboxes';

  /**
   * {@inheritdoc}
   */
  public function answeringForm(array &$form, FormStateInterface $form_state, Answer $answer): void {
    parent::answeringForm($form, $form_state, $answer);
    $this->addFeedbackElementsToAnsweringForm($form, $form_state, $answer);
  }

  /**
   * {@inheritdoc}
   */
  private function buildFeedbackRenderArray(bool $is_correct, Answer $answer, ActivityInterface $activity): array {
    $feedback = [];
    $activity = $answer->getActivity();
    $feedback_field = $is_correct ? 'feedback_if_correct' : 'feedback_if_wrong';
    $feedback_field_item_list = $activity->get($feedback_field);

    if (!$feedback_field_item_list->isEmpty()) {
      $feedback['value'] = [
        '#markup' => $feedback_field_item_list->value,
      ];
      $feedback['#attributes']['class'][] = $is_correct ? self::CLASS_CORRECT : self::CLASS_WRONG;
    }

    return $feedback;
  }

  /**
   * {@inheritdoc}
   */
  private function addAnswerClassesToForm(array &$form, bool $is_correct, Answer $answer): void {
    $activity = $answer->getActivity();
    $current_answer = $answer->getData()['answer'] ?? [];

    foreach ($activity->get('answers') as $delta => $answer_item) {
      /** @var \Drupal\lms_answer_plugins\Plugin\Field\FieldType\LmsAnswer $answer_item */
      $should_be_selected = $answer_item->isCorrect();
      $is_selected = \array_key_exists($delta, $current_answer) && $current_answer[$delta] !== 0;
      $is_option_correct = ($should_be_selected === $is_selected);
      $option_class = $is_option_correct ? self::CLASS_CORRECT : self::CLASS_WRONG;
      $form['answer_feedback_wrapper']['answer'][$delta]['#wrapper_attributes']['class'][] = $option_class;
    }
  }

}
