<?php

declare(strict_types=1);

namespace Drupal\lms_classes\Access;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\flexible_permissions\CalculatedPermissionsItem;
use Drupal\flexible_permissions\PermissionCalculatorBase;
use Drupal\flexible_permissions\RefinableCalculatedPermissionsInterface;
use Drupal\group\Entity\GroupMembershipInterface;
use Drupal\group\Entity\GroupRelationshipInterface;
use Drupal\group\PermissionScopeInterface;
use Drupal\lms\Entity\Bundle\Course;
use Drupal\lms_classes\ClassHelper;

/**
 * Calculates inherited group permissions for an account.
 */
final class ClassPermissionCalculator extends PermissionCalculatorBase {

  /**
   * Course -> class group permission mapping.
   */
  private const PERMISSION_MAPPING = [
    'add students' => 'administer members',
    'view students' => 'view group_membership relationship',
  ];

  /**
   * The constructor.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function calculatePermissions(AccountInterface $account, $scope) {
    $calculated_permissions = parent::calculatePermissions($account, $scope);
    \assert($calculated_permissions instanceof RefinableCalculatedPermissionsInterface);

    if ($scope !== PermissionScopeInterface::INDIVIDUAL_ID) {
      return $calculated_permissions;
    }

    // The inherited permissions need to be recalculated whenever the user is
    // added to or removed from a group.
    $calculated_permissions->addCacheTags(['group_relationship_list:plugin:group_membership:entity:' . $account->id()]);

    // 1. Course members should always be able to see classes
    //    and their members.
    $memberships = $this->entityTypeManager->getStorage('group_relationship')->loadByProperties([
      'plugin_id' => 'group_membership',
      'entity_id' => $account->id(),
      'group_type' => 'lms_course',
    ]);
    $class_ids = [];
    foreach ($memberships as $membership) {
      $calculated_permissions->addCacheableDependency($membership);

      \assert($membership instanceof GroupMembershipInterface);
      $course = $membership->getGroup();
      \assert($course instanceof Course);

      $class_permissions = ['view group' => 'view group'];
      foreach ($membership->getRoles(TRUE) as $role) {
        $calculated_permissions->addCacheableDependency($role);
        foreach (self::PERMISSION_MAPPING as $course_permission => $class_permission) {
          if ($role->hasPermission($course_permission)) {
            $class_permissions[$class_permission] = $class_permission;
          }
        }
      }

      foreach (ClassHelper::getClasses($course) as $class) {
        $class_id = $class->id();
        if (\array_key_exists($class_id, $class_ids)) {
          continue;
        }
        $class_ids[$class_id] = TRUE;

        $calculated_permissions->addItem(new CalculatedPermissionsItem(
          $scope,
          $class_id,
          $class_permissions,
          FALSE,
        ));
      }
    }

    // 2. Class members should always have 'view group' and 'take course'
    //    permissions on the parent course group.
    $memberships = $this->entityTypeManager->getStorage('group_relationship')->loadByProperties([
      'plugin_id' => 'group_membership',
      'entity_id' => $account->id(),
      'group_type' => 'lms_class',
    ]);
    $courses = [];

    foreach ($memberships as $membership) {
      \assert($membership instanceof GroupMembershipInterface);
      $course_relationships = $this->entityTypeManager->getStorage('group_relationship')->loadByProperties([
        'plugin_id' => 'lms_classes',
        'entity_id' => $membership->getGroupId(),
        'group_type' => 'lms_course',
      ]);

      $has_courses = FALSE;
      foreach ($course_relationships as $relationship) {
        \assert($relationship instanceof GroupRelationshipInterface);
        $course = $relationship->getGroup();
        if (!$course instanceof Course) {
          continue;
        }
        $has_courses = TRUE;
        if (!\array_key_exists($course->id(), $courses)) {
          $courses[$course->id()] = $course;
          $calculated_permissions->addCacheableDependency($course);
        }
      }

      if ($has_courses) {
        $calculated_permissions->addCacheableDependency($membership);
      }

      foreach ($courses as $course) {
        $calculated_permissions->addItem(new CalculatedPermissionsItem(
          $scope,
          $course->id(),
          ['take course', 'view group'],
          FALSE
        ));
      }
    }

    return $calculated_permissions;
  }

  /**
   * {@inheritdoc}
   */
  public function getPersistentCacheContexts($scope) {
    if ($scope === PermissionScopeInterface::INDIVIDUAL_ID) {
      return ['user'];
    }
    return [];
  }

}
