<?php

declare(strict_types=1);

namespace Drupal\lms_certificate_entity\Controller;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Controller\ControllerResolverInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\dynamic_entity_reference\Plugin\Field\FieldType\DynamicEntityReferenceFieldItemList;
use Drupal\group\Entity\GroupInterface;
use Drupal\lms\Entity\CourseStatusInterface;
use Drupal\lms\TrainingManager;
use Drupal\lms_certificate_entity\Entity\LmsCertificateEntity;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Handles certificates for courses.
 *
 * @todo move this logic to a controller in the main lms_certificate module, and
 * the fillpdf-specific logic to a plugin or tagged service implementation
 * called by that controller.
 * @see https://www.drupal.org/project/lms_certificate/issues/3502791
 */
class CertificateController extends ControllerBase {

  public function __construct(
    protected ControllerResolverInterface $controllerResolver,
    protected TrainingManager $trainingManager,
    protected EntityTypeManagerInterface $entityTypeMgr,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('controller_resolver'),
      $container->get('lms.training_manager'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * Generates a certificate given a group.
   */
  public function getCertificate(GroupInterface $group) {
    $course_status = $this->trainingManager->loadCourseStatus($group, $this->currentUser(), ['status' => CourseStatusInterface::STATUS_PASSED]);
    if (empty($course_status)) {
      // Shouldn't happen due to the access() below, but let's cover our bases.
      throw new NotFoundHttpException();
    }

    if (!$group->hasField('certificate')) {
      // If there's no field, something is really off, so let
      throw new NotFoundHttpException();
    }

    $certificates_field = $group->get('certificate');
    if (empty($certificates_field) || !($certificates_field instanceof DynamicEntityReferenceFieldItemList)) {
      throw new NotFoundHttpException();
    }

    $certifcates = $group->get('certificate')->referencedEntities();
    if (empty($certifcates)) {
      throw new NotFoundHttpException();
    }

    $certificate = reset($certifcates);
    if (empty($certificate) || !($certificate instanceof LmsCertificateEntity)) {
      throw new NotFoundHttpException();
    }

    // Got a cert? Render it!
    $view_builder = $this->entityTypeMgr->getViewBuilder('lms_certificate_entity');
    $display_mode = $this->config('lms_certificate_entity.settings')->get('display_mode') ?: 'default';
    return $view_builder->view($certificate, $display_mode);
  }

  /**
   * Access callback.
   */
  public function access(GroupInterface $group): AccessResult {

    if ($group->get('certificate')->isEmpty()) {
      $access_result = AccessResult::forbidden()->addCacheableDependency($group);
      return $access_result;
    }
    $course_status = $this->trainingManager->loadCourseStatus($group, $this->currentUser(), ['status' => CourseStatusInterface::STATUS_PASSED]);

    $access_result = AccessResult::allowedIf($course_status !== NULL)
      ->addCacheableDependency($group)
      ->cachePerUser();

    if ($course_status) {
      // If the user has passed, cache the result - the situation can only
      // change if the course status changes again to a non-passed state.
      $access_result->addCacheableDependency($course_status);
    }
    else {
      $access_result->setCacheMaxAge(0);
    }
    // Set to cache per user even though the resulting response may be
    // uncacheable.
    $access_result->cachePerUser();
    return $access_result;
  }

}
