<?php

declare(strict_types=1);

namespace Drupal\lms_certificate_entity\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Form\DeleteMultipleForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\lms_certificate_entity\Form\LmsCertificateEntityForm;
use Drupal\lms_certificate_entity\LmsCertificateEntityAccessControlHandler;
use Drupal\lms_certificate_entity\LmsCertificateEntityInterface;
use Drupal\lms_certificate_entity\LmsCertificateEntityListBuilder;
use Drupal\views\EntityViewsData;

/**
 * Defines the lms_certificate_entity entity class.
 */
#[ContentEntityType(
  id: 'lms_certificate_entity',
  label: new TranslatableMarkup('Certificate'),
  label_collection: new TranslatableMarkup('Certificates'),
  label_singular: new TranslatableMarkup('Certificate'),
  label_plural: new TranslatableMarkup('Certificates'),
  entity_keys: [
    'id' => 'id',
    'langcode' => 'langcode',
    'bundle' => 'bundle',
    'label' => 'label',
    'published' => 'status',
    'uuid' => 'uuid',
  ],
  handlers: [
    'list_builder' => LmsCertificateEntityListBuilder::class,
    'views_data' => EntityViewsData::class,
    'access' => LmsCertificateEntityAccessControlHandler::class,
    'form' => [
      'add' => LmsCertificateEntityForm::class,
      'edit' => LmsCertificateEntityForm::class,
      'delete' => ContentEntityDeleteForm::class,
      'delete-multiple-confirm' => DeleteMultipleForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  links: [
    'collection' => '/admin/content/lms-certificate',
    'add-form' => '/admin/content/lms-certificate/add/{lms_certificate_entity_type}',
    'add-page' => '/admin/content/lms-certificate/add',
    'canonical' => '/certificates/{lms_certificate_entity}',
    'edit-form' => '/certificates/{lms_certificate_entity}/edit',
    'delete-form' => '/certificates/{lms_certificate_entity}/delete',
    'delete-multiple-form' => '/admin/content/lms-certificate-entity/delete-multiple',
  ],
  admin_permission: 'administer lms_certificate_entity types',
  bundle_entity_type: 'lms_certificate_entity_type',
  bundle_label: new TranslatableMarkup('lms_certificate_entity type'),
  base_table: 'lms_certificate_entity',
  data_table: 'lms_certificate_entity_field_data',
  translatable: TRUE,
  label_count: [
    'singular' => '@count certificates',
    'plural' => '@count certificates',
  ],
  field_ui_base_route: 'entity.lms_certificate_entity_type.edit_form',
)]
class LmsCertificateEntity extends ContentEntityBase implements LmsCertificateEntityInterface {

  use EntityChangedTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setTranslatable(TRUE)
      ->setLabel(t('Label'))
      ->setRequired(TRUE)
      ->setSetting('max_length', 255)
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -5,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the lms_certificate_entity was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setTranslatable(TRUE)
      ->setDescription(t('The time that the lms_certificate_entity was last edited.'));

    return $fields;
  }

}
