<?php

declare(strict_types=1);

namespace Drupal\lms_certificate_entity\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure lms_certificate_entity settings for lms_certificate_entity.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * Builds the Settings Form.
   */
  public function __construct(ConfigFactoryInterface $config_factory,
                              TypedConfigManagerInterface $typedConfigManager,
                              protected readonly EntityDisplayRepositoryInterface $entityDisplayRepository) {

    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_display.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'lms_certificate_entity_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['lms_certificate_entity.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['display_mode'] = [

      '#type' => 'select',
      '#title' => $this->t('Display mode'),
      '#description' => $this->t('The display mode to use when displaying the certificate.'),
      '#default_value' => $this->config('lms_certificate_entity.settings')->get('display_mode') ?: 'default',
      '#options' => $this->buildOptions(),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('lms_certificate_entity.settings')
      ->set('display_mode', $form_state->getValue('display_mode'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Build a list of display options for the LMS Certificate Entity.
   *
   * @return array
   *  An array of Display Options for certificates, keyed by display ID.
   */
  protected function buildOptions() : array {
    $displays = $this->entityDisplayRepository->getViewModeOptions('lms_certificate_entity');
    $options = [];
    foreach ($displays as $key => $label) {
      $options[$key] = strval($label);
    }

    return $options;
  }

}
